<?php

namespace App\Models;

use CodeIgniter\Model;

class UserModel extends Model
{
    protected $table            = 'users';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'id',
        'employee_id',
        'username',
        'email',
        'password_hash',
        'first_name',
        'last_name',
        'full_name',
        'role_id',
        'location_id',
        'phone',
        'profile_photo',
        'photo',
        'is_active',
        'status',
        'last_login',
        'password_reset_token',
        'password_reset_expires',
        'remember_token'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'username'   => 'required|max_length[100]|is_unique[users.username,id,{id}]',
        'email'      => 'required|valid_email|max_length[255]|is_unique[users.email,id,{id}]',
        'role_id'    => 'required|is_natural_no_zero',
    ];

    protected $validationMessages = [
        'username' => [
            'required'  => 'Username is required',
            'is_unique' => 'This username is already taken',
        ],
        'email' => [
            'required'    => 'Email is required',
            'valid_email' => 'Please enter a valid email address',
            'is_unique'   => 'This email is already registered',
        ],
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = ['hashPassword'];
    protected $beforeUpdate   = ['hashPassword'];

    /**
     * Hash password before insert/update
     */
    protected function hashPassword(array $data)
    {
        if (isset($data['data']['password_hash']) && !empty($data['data']['password_hash'])) {
            $data['data']['password_hash'] = password_hash($data['data']['password_hash'], PASSWORD_BCRYPT);
        }
        return $data;
    }

    /**
     * Get user with role and permissions
     */
    public function getUserWithRole($userId)
    {
        $user = $this->select('users.*, roles.name as role_name, roles.slug as role_slug, locations.name as location_name')
            ->join('roles', 'roles.id = users.role_id', 'left')
            ->join('locations', 'locations.id = users.location_id', 'left')
            ->find($userId);

        if (!$user) {
            return null;
        }

        // Get permissions
        $permissions = $this->db->table('role_permissions')
            ->select('permissions.slug as permission_slug')
            ->join('permissions', 'permissions.id = role_permissions.permission_id')
            ->where('role_permissions.role_id', $user['role_id'])
            ->get()
            ->getResultArray();

        $user['permissions'] = array_column($permissions, 'permission_slug');

        return $user;
    }

    /**
     * Verify user credentials
     */
    public function verifyCredentials($username, $password)
    {
        $user = $this->where('username', $username)
            ->orWhere('email', $username)
            ->first();

        if (!$user) {
            return false;
        }

        if (!password_verify($password, $user['password_hash'])) {
            return false;
        }

        if ($user['status'] != 'active') {
            return false;
        }

        return $user;
    }

    /**
     * Update last login
     */
    public function updateLastLogin($userId)
    {
        return $this->update($userId, ['last_login' => date('Y-m-d H:i:s')]);
    }

    /**
     * Check if user has permission
     */
    public function hasPermission($userId, $permissionSlug)
    {
        $result = $this->db->table('users')
            ->select('permissions.id')
            ->join('role_permissions', 'role_permissions.role_id = users.role_id')
            ->join('permissions', 'permissions.id = role_permissions.permission_id')
            ->where('users.id', $userId)
            ->where('permissions.slug', $permissionSlug)
            ->get()
            ->getRowArray();

        return !empty($result);
    }
}
