<?php

namespace App\Models;

use CodeIgniter\Model;

class SyncLogModel extends Model
{
    protected $table            = 'sync_logs';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'location_id',
        'sync_type',
        'records_synced',
        'status',
        'started_at',
        'completed_at',
        'error_message'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';

    /**
     * Get latest sync status for each location
     */
    public function getLatestSyncStatus()
    {
        return $this->select('sync_logs.*,
                locations.name as location_name,
                locations.code as location_code')
            ->join('locations', 'locations.id = sync_logs.location_id')
            ->groupBy('sync_logs.location_id')
            ->orderBy('sync_logs.created_at', 'DESC')
            ->findAll();
    }

    /**
     * Get sync history for a location
     */
    public function getLocationSyncHistory($locationId, $limit = 50)
    {
        return $this->where('location_id', $locationId)
            ->orderBy('created_at', 'DESC')
            ->limit($limit)
            ->findAll();
    }

    /**
     * Start sync log
     */
    public function startSync($locationId, $syncType, $syncDirection)
    {
        $data = [
            'location_id'    => $locationId,
            'sync_type'      => $syncType,
            'sync_direction' => $syncDirection,
            'sync_status'    => 'In Progress',
            'started_at'     => date('Y-m-d H:i:s'),
            'created_at'     => date('Y-m-d H:i:s'),
        ];

        $this->insert($data);
        return $this->getInsertID();
    }

    /**
     * Complete sync log
     */
    public function completeSync($syncId, $status, $recordsSynced = 0, $recordsFailed = 0, $conflictsDetected = 0, $errorMessage = null, $details = null)
    {
        return $this->update($syncId, [
            'sync_status'        => $status,
            'records_synced'     => $recordsSynced,
            'records_failed'     => $recordsFailed,
            'conflicts_detected' => $conflictsDetected,
            'error_message'      => $errorMessage,
            'sync_details'       => $details ? json_encode($details) : null,
            'completed_at'       => date('Y-m-d H:i:s'),
        ]);
    }
}
