<?php

namespace App\Models;

use CodeIgniter\Model;

class RoleModel extends Model
{
    protected $table            = 'roles';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'name',
        'slug',
        'description',
        'is_active'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'name' => 'required|max_length[50]|is_unique[roles.name,id,{id}]',
        'slug' => 'required|max_length[50]|is_unique[roles.slug,id,{id}]|alpha_dash',
    ];

    protected $validationMessages = [
        'name' => [
            'required'  => 'Role name is required',
            'is_unique' => 'This role name already exists',
        ],
        'slug' => [
            'required'   => 'Role slug is required',
            'is_unique'  => 'This role slug already exists',
            'alpha_dash' => 'Role slug can only contain alphanumeric characters, underscores and dashes',
        ],
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get role with permissions
     */
    public function getRoleWithPermissions($roleId)
    {
        $role = $this->find($roleId);
        if (!$role) {
            return null;
        }

        $permissions = $this->db->table('role_permissions')
            ->select('permissions.*')
            ->join('permissions', 'permissions.id = role_permissions.permission_id')
            ->where('role_permissions.role_id', $roleId)
            ->get()
            ->getResultArray();

        $role['permissions'] = $permissions;
        return $role;
    }

    /**
     * Assign permissions to role
     */
    public function assignPermissions($roleId, $permissionIds)
    {
        // Remove existing permissions
        $this->db->table('role_permissions')
            ->where('role_id', $roleId)
            ->delete();

        // Add new permissions
        if (!empty($permissionIds)) {
            $data = [];
            foreach ($permissionIds as $permissionId) {
                $data[] = [
                    'role_id'       => $roleId,
                    'permission_id' => $permissionId,
                    'created_at'    => date('Y-m-d H:i:s'),
                ];
            }
            $this->db->table('role_permissions')->insertBatch($data);
        }

        return true;
    }
}
