<?php

namespace App\Models;

use CodeIgniter\Model;

class LocationModel extends Model
{
    protected $table            = 'locations';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'name',
        'code',
        'address',
        'city',
        'province',
        'contact_number',
        'email',
        'timezone',
        'latitude',
        'longitude',
        'is_active'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'code' => 'required|max_length[20]|is_unique[locations.code,id,{id}]',
        'name' => 'required|max_length[100]',
        'email' => 'permit_empty|valid_email',
        'timezone' => 'permit_empty|max_length[50]',
    ];

    protected $validationMessages = [
        'code' => [
            'required'  => 'Location code is required',
            'is_unique' => 'This location code already exists',
        ],
        'name' => [
            'required' => 'Location name is required',
        ],
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert   = [];
    protected $afterInsert    = [];
    protected $beforeUpdate   = [];
    protected $afterUpdate    = [];
    protected $beforeFind     = [];
    protected $afterFind      = [];
    protected $beforeDelete   = [];
    protected $afterDelete    = [];

    /**
     * Get active locations
     */
    public function getActiveLocations()
    {
        return $this->where('is_active', 1)->findAll();
    }

    /**
     * Get location with statistics
     */
    public function getLocationWithStats($id)
    {
        return $this->db->table($this->table)
            ->select('locations.*,
                COUNT(DISTINCT employees.employee_id) as total_employees,
                COUNT(DISTINCT CASE WHEN attendance.attendance_date = CURDATE() THEN attendance.employee_id END) as today_attendance')
            ->join('employees', 'employees.location_id = locations.id', 'left')
            ->join('attendance', 'attendance.employee_id = employees.employee_id', 'left')
            ->where('locations.id', $id)
            ->groupBy('locations.id')
            ->get()
            ->getRowArray();
    }
}
