<?php

namespace App\Models;

use CodeIgniter\Model;

class LeaveRequestModel extends Model
{
    protected $table            = 'leave_requests';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'employee_id',
        'leave_type_id',
        'start_date',
        'end_date',
        'total_days',
        'reason',
        'attachment',
        'status',
        'approved_by',
        'approved_at',
        'rejection_reason'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';

    // Validation
    protected $validationRules = [
        'employee_id'   => 'required|is_natural_no_zero',
        'leave_type_id' => 'required|is_natural_no_zero',
        'start_date'    => 'required|valid_date',
        'end_date'      => 'required|valid_date',
        'total_days'    => 'required|decimal',
        'reason'        => 'required',
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get leave requests with employee details
     */
    public function getRequestsWithDetails($filters = [])
    {
        $builder = $this->select('leave_requests.*,
                employees.employee_number as emp_id,
                employees.first_name,
                employees.last_name,
                employees.photo,
                employees.department_id,
                leave_types.name as leave_type_name,
                leave_types.color,
                departments.name as department_name,
                locations.name as location_name,
                users.username as approved_by_name')
            ->join('employees', 'employees.employee_id = leave_requests.employee_id')
            ->join('leave_types', 'leave_types.id = leave_requests.leave_type_id')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = employees.location_id', 'left')
            ->join('users', 'users.id = leave_requests.approved_by', 'left');

        if (!empty($filters['status'])) {
            $builder->where('leave_requests.status', $filters['status']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('leave_requests.employee_id', $filters['employee_id']);
        }

        if (!empty($filters['location_id'])) {
            $builder->where('employees.location_id', $filters['location_id']);
        }

        if (!empty($filters['leave_type_id'])) {
            $builder->where('leave_requests.leave_type_id', $filters['leave_type_id']);
        }

        if (!empty($filters['date_from'])) {
            $builder->where('leave_requests.start_date >=', $filters['date_from']);
        }

        if (!empty($filters['date_to'])) {
            $builder->where('leave_requests.end_date <=', $filters['date_to']);
        }

        return $builder->orderBy('leave_requests.created_at', 'DESC')->findAll();
    }

    /**
     * Approve leave request
     */
    public function approveRequest($requestId, $approvedBy, $comments = null)
    {
        $request = $this->find($requestId);
        if (!$request || $request['status'] !== 'Pending') {
            return false;
        }

        $this->transStart();

        // Update request status
        $this->update($requestId, [
            'status'      => 'Approved',
            'approved_by' => $approvedBy,
            'approved_at' => date('Y-m-d H:i:s'),
            'comments'    => $comments,
        ]);

        // Update leave balance
        $leaveBalanceModel = new LeaveBalanceModel();
        $leaveBalanceModel->updateBalance(
            $request['employee_id'],
            $request['leave_type_id'],
            $request['total_days'],
            'use'
        );

        $this->transComplete();

        return $this->transStatus();
    }

    /**
     * Reject leave request
     */
    public function rejectRequest($requestId, $approvedBy, $rejectionReason)
    {
        $request = $this->find($requestId);
        if (!$request || $request['status'] !== 'Pending') {
            return false;
        }

        return $this->update($requestId, [
            'status'           => 'Rejected',
            'approved_by'      => $approvedBy,
            'approved_at'      => date('Y-m-d H:i:s'),
            'rejection_reason' => $rejectionReason,
        ]);
    }

    /**
     * Get leave calendar data
     */
    public function getLeaveCalendar($startDate, $endDate, $locationId = null)
    {
        $builder = $this->select('leave_requests.*,
                employees.employee_id as emp_id,
                employees.first_name,
                employees.last_name,
                leave_types.leave_type_name,
                leave_types.color_code')
            ->join('employees', 'employees.employee_id = leave_requests.employee_id')
            ->join('leave_types', 'leave_types.id = leave_requests.leave_type_id')
            ->where('leave_requests.status', 'Approved')
            ->groupStart()
                ->where('leave_requests.start_date >=', $startDate)
                ->where('leave_requests.start_date <=', $endDate)
            ->groupEnd()
            ->orGroupStart()
                ->where('leave_requests.end_date >=', $startDate)
                ->where('leave_requests.end_date <=', $endDate)
            ->groupEnd();

        if ($locationId) {
            $builder->where('employees.location_id', $locationId);
        }

        return $builder->findAll();
    }

    /**
     * Get pending requests count
     */
    public function getPendingCount($locationId = null)
    {
        $builder = $this->where('status', 'Pending');

        if ($locationId) {
            $builder->join('employees', 'employees.employee_id = leave_requests.employee_id')
                ->where('employees.location_id', $locationId);
        }

        return $builder->countAllResults();
    }
}
