<?php

namespace App\Models;

use CodeIgniter\Model;

class LeaveBalanceModel extends Model
{
    protected $table            = 'leave_balances';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = false;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'employee_id',
        'leave_type_id',
        'year',
        'total_days',
        'used_days',
        'remaining_days'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';

    // Validation
    protected $validationRules = [
        'employee_id'   => 'required|is_natural_no_zero',
        'leave_type_id' => 'required|is_natural_no_zero',
        'year'          => 'required|is_natural_no_zero',
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get employee leave balances for current year
     */
    public function getEmployeeBalances($employeeId, $year = null)
    {
        $year = $year ?? date('Y');

        return $this->select('leave_balances.*,
                leave_types.name as leave_type_name,
                leave_types.code as leave_type_code,
                leave_types.color')
            ->join('leave_types', 'leave_types.id = leave_balances.leave_type_id')
            ->where('leave_balances.employee_id', $employeeId)
            ->where('leave_balances.year', $year)
            ->findAll();
    }

    /**
     * Update leave balance after approval
     */
    public function updateBalance($employeeId, $leaveTypeId, $days, $action = 'use')
    {
        $year = date('Y');
        $balance = $this->where('employee_id', $employeeId)
            ->where('leave_type_id', $leaveTypeId)
            ->where('year', $year)
            ->first();

        if (!$balance) {
            return false;
        }

        if ($action === 'use') {
            $newUsedDays = $balance['used_days'] + $days;
            $newRemainingDays = $balance['remaining_days'] - $days;
        } else {
            $newUsedDays = $balance['used_days'] - $days;
            $newRemainingDays = $balance['remaining_days'] + $days;
        }

        return $this->update($balance['id'], [
            'used_days'      => $newUsedDays,
            'remaining_days' => $newRemainingDays,
        ]);
    }

    /**
     * Initialize leave balances for new employee
     */
    public function initializeEmployeeBalances($employeeId, $year = null)
    {
        $year = $year ?? date('Y');
        $leaveTypeModel = new LeaveTypeModel();
        $leaveTypes = $leaveTypeModel->getActiveLeaveTypes();

        foreach ($leaveTypes as $leaveType) {
            $this->insert([
                'employee_id'   => $employeeId,
                'leave_type_id' => $leaveType['id'],
                'year'          => $year,
                'total_days'    => $leaveType['default_days_per_year'],
                'remaining_days' => $leaveType['default_days_per_year'],
            ]);
        }

        return true;
    }
}
