<?php

namespace App\Models;

use CodeIgniter\Model;

class AttendanceModel extends Model
{
    protected $table            = 'attendance';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;
    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;
    protected $protectFields    = true;
    protected $allowedFields    = [
        'employee_id',
        'attendance_date',
        'sign_in_time',
        'lunch_out_time',
        'lunch_in_time',
        'lunch_out_method',
        'lunch_in_method',
        'lunch_duration_minutes',
        'sign_out_time',
        'sign_in_method',
        'sign_out_method',
        'status',
        'is_late',
        'late_minutes',
        'working_hours',
        'overtime_hours',
        'notes',
        'location_id',
        'created_by',
        'local_uuid',
        'device_time'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat    = 'datetime';
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $deletedField  = 'deleted_at';

    // Validation
    protected $validationRules = [
        'employee_id'     => 'required|is_natural_no_zero',
        'location_id'     => 'required|is_natural_no_zero',
        'attendance_date' => 'required|valid_date',
    ];

    protected $skipValidation       = false;
    protected $cleanValidationRules = true;

    /**
     * Get today's attendance for all employees
     */
    public function getTodayAttendance($locationId = null)
    {
        $builder = $this->select('attendance.*,
                employees.employee_number as emp_id,
                employees.first_name,
                employees.last_name,
                employees.photo,
                employees.department_id,
                departments.name as department_name,
                locations.name as location_name')
            ->join('employees', 'employees.employee_id = attendance.employee_id')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = attendance.location_id')
            ->where('attendance.attendance_date', date('Y-m-d'));

        if ($locationId) {
            $builder->where('attendance.location_id', $locationId);
        }

        return $builder->findAll();
    }

    /**
     * Get attendance by date range
     */
    public function getAttendanceByDateRange($startDate, $endDate, $filters = [])
    {
        $builder = $this->select('attendance.*,
                employees.employee_number as emp_id,
                employees.first_name,
                employees.last_name,
                departments.name as department_name,
                locations.name as location_name')
            ->join('employees', 'employees.employee_id = attendance.employee_id')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = attendance.location_id')
            ->where('attendance.attendance_date >=', $startDate)
            ->where('attendance.attendance_date <=', $endDate);

        if (!empty($filters['location_id'])) {
            $builder->where('attendance.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('employees.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('attendance.employee_id', $filters['employee_id']);
        }

        if (!empty($filters['status'])) {
            $builder->where('attendance.status', $filters['status']);
        }

        return $builder->orderBy('attendance.attendance_date', 'DESC')->findAll();
    }

    /**
     * Get attendance statistics for dashboard
     */
    public function getAttendanceStats($date = null, $locationId = null)
    {
        $date = $date ?? date('Y-m-d');

        $builder = $this->db->table($this->table)
            ->select('
                COUNT(*) as total_records,
                SUM(CASE WHEN status = "present" THEN 1 ELSE 0 END) as present_count,
                SUM(CASE WHEN status = "late" THEN 1 ELSE 0 END) as late_count,
                SUM(CASE WHEN status = "absent" THEN 1 ELSE 0 END) as absent_count,
                SUM(CASE WHEN status = "on-leave" THEN 1 ELSE 0 END) as on_leave_count,
                AVG(working_hours) as avg_working_hours,
                SUM(late_minutes) as total_late_minutes,
                SUM(overtime_hours) as total_overtime_hours
            ')
            ->where('attendance_date', $date);

        if ($locationId) {
            $builder->where('location_id', $locationId);
        }

        return $builder->get()->getRowArray();
    }

    /**
     * Get employee attendance summary
     */
    public function getEmployeeAttendanceSummary($employeeId, $month = null, $year = null)
    {
        $month = $month ?? date('m');
        $year = $year ?? date('Y');

        return $this->db->table($this->table)
            ->select('
                COUNT(*) as total_days,
                SUM(CASE WHEN status = "Present" OR status = "Late" THEN 1 ELSE 0 END) as present_days,
                SUM(CASE WHEN status = "Late" THEN 1 ELSE 0 END) as late_days,
                SUM(CASE WHEN status = "Absent" THEN 1 ELSE 0 END) as absent_days,
                SUM(CASE WHEN status = "On Leave" THEN 1 ELSE 0 END) as leave_days,
                SUM(working_hours) as total_hours,
                SUM(late_minutes) as total_late_minutes,
                AVG(working_hours) as avg_hours_per_day
            ')
            ->where('employee_id', $employeeId)
            ->where('MONTH(attendance_date)', $month)
            ->where('YEAR(attendance_date)', $year)
            ->get()
            ->getRowArray();
    }

    /**
     * Record sign in
     */
    public function recordSignIn($data)
    {
        $attendanceData = [
            'employee_id'       => $data['employee_id'],
            'location_id'       => $data['location_id'],
            'attendance_date'   => date('Y-m-d'),
            'sign_in_time'      => date('Y-m-d H:i:s'),
            'sign_in_method'    => $data['method'] ?? 'QR Code',
            'sign_in_device_id' => $data['device_id'] ?? null,
            'sign_in_photo'     => $data['photo'] ?? null,
            'status'            => 'Present',
        ];

        // Check if already signed in today
        $existing = $this->where('employee_id', $data['employee_id'])
            ->where('attendance_date', date('Y-m-d'))
            ->first();

        if ($existing) {
            return ['success' => false, 'message' => 'Already signed in today'];
        }

        if ($this->insert($attendanceData)) {
            return ['success' => true, 'id' => $this->getInsertID()];
        }

        return ['success' => false, 'message' => 'Failed to record sign in'];
    }

    /**
     * Record sign out
     */
    public function recordSignOut($employeeId, $data)
    {
        $attendance = $this->where('employee_id', $employeeId)
            ->where('attendance_date', date('Y-m-d'))
            ->first();

        if (!$attendance) {
            return ['success' => false, 'message' => 'No sign in record found for today'];
        }

        if ($attendance['sign_out_time']) {
            return ['success' => false, 'message' => 'Already signed out today'];
        }

        $updateData = [
            'sign_out_time'      => date('Y-m-d H:i:s'),
            'sign_out_method'    => $data['method'] ?? 'QR Code',
            'sign_out_device_id' => $data['device_id'] ?? null,
            'sign_out_photo'     => $data['photo'] ?? null,
        ];

        // Calculate working hours
        $signIn = strtotime($attendance['sign_in_time']);
        $signOut = time();
        $workingHours = round(($signOut - $signIn) / 3600, 2);
        $updateData['working_hours'] = $workingHours;

        if ($this->update($attendance['id'], $updateData)) {
            return ['success' => true];
        }

        return ['success' => false, 'message' => 'Failed to record sign out'];
    }

    /**
     * Get pending sync records
     */
    public function getPendingSyncRecords($locationId)
    {
        return $this->where('location_id', $locationId)
            ->where('sync_status', 'Pending')
            ->findAll();
    }
}
