<?php

if (!function_exists('is_logged_in')) {
    /**
     * Check if user is logged in
     *
     * @return bool
     */
    function is_logged_in(): bool
    {
        return session()->get('isLoggedIn') === true;
    }
}

if (!function_exists('current_user')) {
    /**
     * Get current logged in user data
     *
     * @param string|null $key Specific key to retrieve
     * @return mixed
     */
    function current_user(?string $key = null)
    {
        if ($key) {
            return session()->get($key);
        }

        return [
            'id'            => session()->get('user_id'),
            'username'      => session()->get('username'),
            'email'         => session()->get('email'),
            'first_name'    => session()->get('first_name'),
            'last_name'     => session()->get('last_name'),
            'full_name'     => session()->get('full_name'),
            'role_id'       => session()->get('role_id'),
            'role_name'     => session()->get('role_name'),
            'role_slug'     => session()->get('role_slug'),
            'location_id'   => session()->get('location_id'),
            'location_name' => session()->get('location_name'),
            'profile_photo' => session()->get('profile_photo'),
        ];
    }
}

if (!function_exists('user_id')) {
    /**
     * Get current user ID
     *
     * @return int|null
     */
    function user_id(): ?int
    {
        return session()->get('user_id');
    }
}

if (!function_exists('has_permission')) {
    /**
     * Check if user has a specific permission
     *
     * @param string|array $permission Permission slug(s)
     * @return bool
     */
    function has_permission($permission): bool
    {
        if (!is_logged_in()) {
            return false;
        }

        // Super admin has all permissions
        if (session()->get('role_slug') === 'super-admin') {
            return true;
        }

        $userPermissions = session()->get('permissions') ?? [];

        // Check for single permission
        if (is_string($permission)) {
            return in_array($permission, $userPermissions);
        }

        // Check for any of multiple permissions
        if (is_array($permission)) {
            foreach ($permission as $perm) {
                if (in_array($perm, $userPermissions)) {
                    return true;
                }
            }
        }

        return false;
    }
}

if (!function_exists('has_all_permissions')) {
    /**
     * Check if user has all specified permissions
     *
     * @param array $permissions Array of permission slugs
     * @return bool
     */
    function has_all_permissions(array $permissions): bool
    {
        if (!is_logged_in()) {
            return false;
        }

        // Super admin has all permissions
        if (session()->get('role_slug') === 'super-admin') {
            return true;
        }

        $userPermissions = session()->get('permissions') ?? [];

        foreach ($permissions as $permission) {
            if (!in_array($permission, $userPermissions)) {
                return false;
            }
        }

        return true;
    }
}

if (!function_exists('has_role')) {
    /**
     * Check if user has a specific role
     *
     * @param string|array $role Role slug(s)
     * @return bool
     */
    function has_role($role): bool
    {
        if (!is_logged_in()) {
            return false;
        }

        $userRole = session()->get('role_slug');

        if (is_string($role)) {
            return $userRole === $role;
        }

        if (is_array($role)) {
            return in_array($userRole, $role);
        }

        return false;
    }
}

if (!function_exists('is_super_admin')) {
    /**
     * Check if current user is super admin
     *
     * @return bool
     */
    function is_super_admin(): bool
    {
        return has_role('super-admin');
    }
}

if (!function_exists('user_location_id')) {
    /**
     * Get current user's location ID
     *
     * @return int|null
     */
    function user_location_id(): ?int
    {
        return session()->get('location_id');
    }
}

if (!function_exists('can_view')) {
    /**
     * Check if user can view a module
     *
     * @param string $module Module name (employees, attendance, etc.)
     * @return bool
     */
    function can_view(string $module): bool
    {
        return has_permission("view-{$module}");
    }
}

if (!function_exists('can_create')) {
    /**
     * Check if user can create in a module
     *
     * @param string $module Module name
     * @return bool
     */
    function can_create(string $module): bool
    {
        return has_permission("create-{$module}");
    }
}

if (!function_exists('can_edit')) {
    /**
     * Check if user can edit in a module
     *
     * @param string $module Module name
     * @return bool
     */
    function can_edit(string $module): bool
    {
        return has_permission("edit-{$module}");
    }
}

if (!function_exists('can_delete')) {
    /**
     * Check if user can delete in a module
     *
     * @param string $module Module name
     * @return bool
     */
    function can_delete(string $module): bool
    {
        return has_permission("delete-{$module}");
    }
}

if (!function_exists('user_avatar')) {
    /**
     * Get user avatar URL
     *
     * @param string|null $photo Profile photo path
     * @return string
     */
    function user_avatar(?string $photo = null): string
    {
        if ($photo && file_exists(FCPATH . $photo)) {
            return base_url($photo);
        }

        // Return default avatar
        return base_url('assets/img/default-avatar.png');
    }
}

if (!function_exists('format_permission')) {
    /**
     * Format permission slug to readable name
     *
     * @param string $slug Permission slug
     * @return string
     */
    function format_permission(string $slug): string
    {
        return ucwords(str_replace(['-', '_'], ' ', $slug));
    }
}
