<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;

class PermissionFilter implements FilterInterface
{
    /**
     * Check if user has required permission
     *
     * @param RequestInterface $request
     * @param array|null $arguments Permission slugs required
     * @return mixed
     */
    public function before(RequestInterface $request, $arguments = null)
    {
        // Check if user is logged in
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login')->with('error', 'Please login to continue');
        }

        // If no specific permissions required, just check authentication
        if (empty($arguments)) {
            return;
        }

        // Get user permissions
        $userPermissions = session()->get('permissions') ?? [];
        $roleSlug = session()->get('role_slug');

        // Super admin has all permissions
        if ($roleSlug === 'super-admin') {
            return;
        }

        // Check if user has any of the required permissions
        $hasPermission = false;
        foreach ($arguments as $permission) {
            if (in_array($permission, $userPermissions)) {
                $hasPermission = true;
                break;
            }
        }

        if (!$hasPermission) {
            // Log unauthorized access attempt
            $auditLogModel = new \App\Models\AuditLogModel();
            $auditLogModel->logAction(
                session()->get('user_id'),
                'Unauthorized Access Attempt',
                'Security',
                null,
                "Attempted to access resource requiring permissions: " . implode(', ', $arguments),
                null,
                null
            );

            return redirect()->to('/dashboard')->with('error', 'You do not have permission to access this resource');
        }
    }

    /**
     * After filter
     *
     * @param RequestInterface $request
     * @param ResponseInterface $response
     * @param array|null $arguments
     * @return mixed
     */
    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        // Do nothing
    }
}
