<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\SystemSettingModel;
use App\Models\AuditLogModel;

class SystemSettingController extends BaseController
{
    protected $systemSettingModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->systemSettingModel = new SystemSettingModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display system settings page
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-system-settings')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view system settings');
        }

        // Get all settings
        $allSettings = $this->systemSettingModel->orderBy('key', 'ASC')->findAll();

        // Group settings by prefix (e.g., app_, work_, email_, etc.)
        $settings = [
            'general' => [],
            'attendance' => [],
            'leave' => [],
            'email' => [],
            'reports' => [],
            'sync' => [],
        ];

        foreach ($allSettings as $setting) {
            $key = $setting['key'];
            // Determine group based on key prefix
            if (str_starts_with($key, 'app_') || str_starts_with($key, 'date_') || str_starts_with($key, 'time_') ||
                str_starts_with($key, 'enable_auto_employee') || str_starts_with($key, 'employee_number')) {
                $settings['general'][] = $setting;
            } elseif (str_starts_with($key, 'work_') || str_starts_with($key, 'grace_') ||
                      str_starts_with($key, 'late_') || str_starts_with($key, 'minimum_') ||
                      str_starts_with($key, 'enable_facial') || str_starts_with($key, 'enable_qr')) {
                $settings['attendance'][] = $setting;
            } elseif (str_starts_with($key, 'leave_') || str_starts_with($key, 'auto_approve') ||
                      str_starts_with($key, 'max_leave')) {
                $settings['leave'][] = $setting;
            } elseif (str_starts_with($key, 'smtp_') || str_starts_with($key, 'email_') ||
                      str_starts_with($key, 'enable_email')) {
                $settings['email'][] = $setting;
            } elseif (str_starts_with($key, 'report_')) {
                $settings['reports'][] = $setting;
            } elseif (str_starts_with($key, 'sync_') || str_starts_with($key, 'enable_offline_sync') ||
                      str_starts_with($key, 'enable_auto_sync') || str_starts_with($key, 'enable_duplicate_detection')) {
                $settings['sync'][] = $setting;
            }
        }

        $data = [
            'pageTitle' => 'System Settings',
            'settings' => $settings,
            'canEdit' => has_permission('edit-system-settings'),
        ];

        return view('settings/system/index', $data);
    }

    /**
     * Update system settings
     */
    public function update()
    {
        // Check permission
        if (!has_permission('edit-system-settings')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to edit system settings');
        }

        $updatedCount = 0;
        $changes = [];

        // Get all posted data
        $postData = $this->request->getPost();

        // Loop through all posted settings
        foreach ($postData as $key => $value) {
            // Skip CSRF token
            if ($key === 'csrf_test_name') {
                continue;
            }

            // Get the setting from database
            $setting = $this->systemSettingModel->where('key', $key)->first();

            if (!$setting) {
                continue;
            }

            // Process value based on type
            $processedValue = $this->processValue($value, $setting['type']);

            // Check if value changed
            if ($setting['value'] != $processedValue) {
                $changes[] = [
                    'key' => $key,
                    'old_value' => $setting['value'],
                    'new_value' => $processedValue,
                ];

                // Update the setting
                $this->systemSettingModel->update($setting['id'], [
                    'value' => $processedValue,
                ]);

                $updatedCount++;
            }
        }

        // Log the changes if any
        if ($updatedCount > 0) {
            $description = "Updated {$updatedCount} system setting(s): " . implode(', ', array_column($changes, 'key'));

            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'System Settings',
                'action' => 'update',
                'record_id' => null,
                'description' => $description,
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return redirect()->to('settings/system')->with('success', "Successfully updated {$updatedCount} setting(s)");
        }

        return redirect()->to('settings/system')->with('info', 'No changes were made');
    }

    /**
     * Process value based on type
     */
    private function processValue($value, $type)
    {
        switch ($type) {
            case 'boolean':
                return $value ? '1' : '0';
            case 'number':
                return (int) $value;
            case 'json':
                return is_array($value) ? json_encode($value) : $value;
            default:
                return $value;
        }
    }

    /**
     * Reset settings to default (AJAX)
     */
    public function reset($group = null)
    {
        // Check permission
        if (!has_permission('edit-system-settings')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to reset settings'
            ]);
        }

        // This would reset settings to their default values
        // For now, just return a message
        return $this->response->setJSON([
            'success' => false,
            'message' => 'Reset functionality not yet implemented. Please manually restore default values.'
        ]);
    }

    /**
     * Test email configuration (AJAX)
     */
    public function testEmail()
    {
        // Check permission
        if (!has_permission('edit-system-settings')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to test email'
            ]);
        }

        $email = \Config\Services::email();

        $testEmail = $this->request->getPost('test_email');
        if (!$testEmail || !filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Please provide a valid email address'
            ]);
        }

        try {
            $email->setTo($testEmail);
            $email->setSubject('LBDI Attendance System - Test Email');
            $email->setMessage('This is a test email from LBDI Attendance Management System. If you received this, your email configuration is working correctly.');

            if ($email->send()) {
                // Log activity
                $this->auditLogModel->insert([
                    'user_id' => current_user('id'),
                    'module' => 'System Settings',
                    'action' => 'test',
                    'record_id' => null,
                    'description' => 'Sent test email to: ' . $testEmail,
                    'ip_address' => $this->request->getIPAddress(),
                    'user_agent' => $this->request->getUserAgent()->getAgentString(),
                ]);

                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Test email sent successfully! Check your inbox.'
                ]);
            } else {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => 'Failed to send test email. Please check your SMTP configuration.',
                    'error' => $email->printDebugger(['headers'])
                ]);
            }
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export settings as JSON
     */
    public function export()
    {
        // Check permission
        if (!has_permission('view-system-settings')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to export settings');
        }

        $db = \Config\Database::connect();
        $settings = $db->table('system_settings')
            ->select('setting_key, setting_value, setting_type, group, description')
            ->get()
            ->getResultArray();

        // Log activity
        $this->auditLogModel->insert([
            'user_id' => current_user('id'),
            'module' => 'System Settings',
            'action' => 'export',
            'record_id' => null,
            'description' => 'Exported system settings',
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ]);

        $filename = 'system_settings_' . date('Y-m-d_His') . '.json';

        return $this->response
            ->setHeader('Content-Type', 'application/json')
            ->setHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->setBody(json_encode($settings, JSON_PRETTY_PRINT));
    }

    /**
     * Clear cache (AJAX)
     */
    public function clearCache()
    {
        // Check permission
        if (!has_permission('edit-system-settings')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to clear cache'
            ]);
        }

        try {
            $cache = \Config\Services::cache();
            $cache->clean();

            // Log activity
            $this->auditLogModel->insert([
                'user_id' => current_user('id'),
                'module' => 'System Settings',
                'action' => 'clear_cache',
                'record_id' => null,
                'description' => 'Cleared system cache',
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
            ]);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Cache cleared successfully'
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to clear cache: ' . $e->getMessage()
            ]);
        }
    }
}
