<?php

namespace App\Controllers\Settings;

use App\Controllers\BaseController;
use App\Models\AuditLogModel;
use App\Models\UserModel;

class AuditLogController extends BaseController
{
    protected $auditLogModel;
    protected $userModel;

    public function __construct()
    {
        $this->auditLogModel = new AuditLogModel();
        $this->userModel = new UserModel();

        helper('auth');
    }

    /**
     * Display audit logs
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-audit-logs')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view audit logs');
        }

        // Get filters from query string
        $filters = [
            'user_id' => $this->request->getGet('user_id'),
            'module' => $this->request->getGet('module'),
            'action' => $this->request->getGet('action'),
            'date_from' => $this->request->getGet('date_from') ?: date('Y-m-d', strtotime('-30 days')),
            'date_to' => $this->request->getGet('date_to') ?: date('Y-m-d'),
            'search' => $this->request->getGet('search'),
        ];

        // Pagination
        $perPage = 50;
        $page = $this->request->getGet('page') ?? 1;

        // Build query
        $builder = $this->auditLogModel->builder();
        $builder->select('audit_logs.*, users.username, users.full_name');
        $builder->join('users', 'users.id = audit_logs.user_id', 'left');

        // Apply filters
        if (!empty($filters['user_id'])) {
            $builder->where('audit_logs.user_id', $filters['user_id']);
        }

        if (!empty($filters['module'])) {
            $builder->where('audit_logs.module', $filters['module']);
        }

        if (!empty($filters['action'])) {
            $builder->where('audit_logs.action', $filters['action']);
        }

        if (!empty($filters['date_from'])) {
            $builder->where('audit_logs.created_at >=', $filters['date_from'] . ' 00:00:00');
        }

        if (!empty($filters['date_to'])) {
            $builder->where('audit_logs.created_at <=', $filters['date_to'] . ' 23:59:59');
        }

        if (!empty($filters['search'])) {
            $builder->groupStart()
                ->like('audit_logs.description', $filters['search'])
                ->orLike('users.username', $filters['search'])
                ->orLike('users.full_name', $filters['search'])
                ->orLike('audit_logs.ip_address', $filters['search'])
                ->groupEnd();
        }

        // Get total count for pagination
        $totalLogs = $builder->countAllResults(false);

        // Get paginated logs
        $builder->orderBy('audit_logs.created_at', 'DESC');
        $builder->limit($perPage, ($page - 1) * $perPage);
        $logs = $builder->get()->getResultArray();

        // Get unique modules and actions for filters
        $modules = $this->auditLogModel->distinct()
            ->select('module')
            ->orderBy('module', 'ASC')
            ->findAll();

        $actions = $this->auditLogModel->distinct()
            ->select('action')
            ->orderBy('action', 'ASC')
            ->findAll();

        // Get all users for filter
        $users = $this->userModel->select('id, username, full_name')
            ->where('status', 'active')
            ->orderBy('full_name', 'ASC')
            ->findAll();

        // Calculate statistics
        $stats = $this->getAuditStatistics($filters);

        $data = [
            'pageTitle' => 'Audit Logs',
            'logs' => $logs,
            'filters' => $filters,
            'modules' => $modules,
            'actions' => $actions,
            'users' => $users,
            'stats' => $stats,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total_logs' => $totalLogs,
                'total_pages' => ceil($totalLogs / $perPage),
            ],
        ];

        return view('settings/audit-logs/index', $data);
    }

    /**
     * Get audit statistics
     */
    private function getAuditStatistics($filters)
    {
        $builder = $this->auditLogModel->builder();

        // Apply date filters
        if (!empty($filters['date_from'])) {
            $builder->where('created_at >=', $filters['date_from'] . ' 00:00:00');
        }
        if (!empty($filters['date_to'])) {
            $builder->where('created_at <=', $filters['date_to'] . ' 23:59:59');
        }

        $totalLogs = $builder->countAllResults(false);

        // Count by action
        $actions = $builder->select('action, COUNT(*) as count')
            ->groupBy('action')
            ->get()
            ->getResultArray();

        $actionCounts = [];
        foreach ($actions as $action) {
            $actionCounts[$action['action']] = $action['count'];
        }

        // Count by module
        $builder = $this->auditLogModel->builder();
        if (!empty($filters['date_from'])) {
            $builder->where('created_at >=', $filters['date_from'] . ' 00:00:00');
        }
        if (!empty($filters['date_to'])) {
            $builder->where('created_at <=', $filters['date_to'] . ' 23:59:59');
        }

        $modules = $builder->select('module, COUNT(*) as count')
            ->groupBy('module')
            ->orderBy('count', 'DESC')
            ->limit(5)
            ->get()
            ->getResultArray();

        return [
            'total' => $totalLogs,
            'actions' => $actionCounts,
            'top_modules' => $modules,
        ];
    }

    /**
     * View log details (AJAX)
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('view-audit-logs')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to view audit logs'
            ]);
        }

        $builder = $this->auditLogModel->builder();
        $builder->select('audit_logs.*, users.username, users.full_name, users.email');
        $builder->join('users', 'users.id = audit_logs.user_id', 'left');
        $builder->where('audit_logs.id', $id);
        $log = $builder->get()->getRowArray();

        if (!$log) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Log not found'
            ]);
        }

        return $this->response->setJSON([
            'success' => true,
            'log' => $log
        ]);
    }

    /**
     * Export audit logs
     */
    public function export()
    {
        // Check permission
        if (!has_permission('view-audit-logs')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to export audit logs');
        }

        // Get filters
        $filters = [
            'user_id' => $this->request->getGet('user_id'),
            'module' => $this->request->getGet('module'),
            'action' => $this->request->getGet('action'),
            'date_from' => $this->request->getGet('date_from') ?: date('Y-m-d', strtotime('-30 days')),
            'date_to' => $this->request->getGet('date_to') ?: date('Y-m-d'),
        ];

        // Get logs
        $logs = $this->auditLogModel->getLogsWithDetails($filters);

        // Prepare CSV
        $csv = "ID,Date/Time,User,Module,Action,Description,IP Address\n";

        foreach ($logs as $log) {
            $username = $log['username'] ?? 'Unknown';
            $fullName = isset($log['first_name']) && isset($log['last_name'])
                ? $log['first_name'] . ' ' . $log['last_name']
                : $username;

            $csv .= sprintf(
                "%d,\"%s\",\"%s\",\"%s\",\"%s\",\"%s\",\"%s\"\n",
                $log['id'],
                $log['created_at'],
                $fullName,
                $log['module'],
                $log['action'],
                str_replace('"', '""', $log['description']),
                $log['ip_address']
            );
        }

        $filename = 'audit_logs_' . date('Y-m-d_His') . '.csv';

        return $this->response
            ->setHeader('Content-Type', 'text/csv')
            ->setHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->setBody($csv);
    }

    /**
     * Clear old logs
     */
    public function clearOld()
    {
        // Check permission
        if (!has_permission('delete-audit-logs')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to delete audit logs'
            ]);
        }

        $days = $this->request->getPost('days') ?? 90;
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        $deleted = $this->auditLogModel->where('created_at <', $cutoffDate)->delete();

        if ($deleted) {
            return $this->response->setJSON([
                'success' => true,
                'message' => "Successfully deleted logs older than {$days} days"
            ]);
        }

        return $this->response->setJSON([
            'success' => false,
            'message' => 'No logs were deleted'
        ]);
    }
}
