<?php

namespace App\Controllers\Reports;

use App\Controllers\BaseController;
use App\Models\AttendanceModel;
use App\Models\EmployeeModel;
use App\Models\DepartmentModel;
use App\Models\LocationModel;
use App\Models\LeaveRequestModel;
use App\Models\AuditLogModel;

class ReportController extends BaseController
{
    protected $attendanceModel;
    protected $employeeModel;
    protected $departmentModel;
    protected $locationModel;
    protected $leaveRequestModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->attendanceModel = new AttendanceModel();
        $this->employeeModel = new EmployeeModel();
        $this->departmentModel = new DepartmentModel();
        $this->locationModel = new LocationModel();
        $this->leaveRequestModel = new LeaveRequestModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Reports Dashboard - Main index page
     */
    public function index()
    {
        // Check permission
        if (!has_permission('generate-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to access reports');
        }

        $db = \Config\Database::connect();
        $today = date('Y-m-d');
        $currentMonth = date('Y-m');

        // Get today's attendance rate
        $totalEmployees = $this->employeeModel->where('status', 'active')->countAllResults();
        $presentToday = $db->table('attendance')
            ->where('attendance_date', $today)
            ->where('status !=', 'absent')
            ->countAllResults();
        $attendanceRate = $totalEmployees > 0 ? round(($presentToday / $totalEmployees) * 100, 1) : 0;

        // Get pending leave requests
        $pendingLeaves = $this->leaveRequestModel
            ->where('status', 'pending')
            ->countAllResults();

        // Get reports generated this month
        $reportsThisMonth = $db->table('audit_logs')
            ->where('action', 'generate')
            ->where('module', 'reports')
            ->where('DATE_FORMAT(created_at, "%Y-%m")', $currentMonth)
            ->countAllResults();

        // Get scheduled reports count
        $scheduledReports = $db->table('scheduled_reports')
            ->where('is_active', 1)
            ->countAllResults(false); // false to not reset query

        // Get recent reports from audit logs
        $recentReports = $db->table('audit_logs')
            ->select('audit_logs.*, users.first_name, users.last_name')
            ->join('users', 'users.id = audit_logs.user_id', 'left')
            ->where('audit_logs.action', 'generate')
            ->where('audit_logs.module', 'reports')
            ->orderBy('audit_logs.created_at', 'DESC')
            ->limit(5)
            ->get()
            ->getResultArray();

        $data = [
            'pageTitle' => 'Reports Dashboard',
            'statistics' => [
                'attendance_rate' => $attendanceRate,
                'pending_leaves' => $pendingLeaves,
                'reports_this_month' => $reportsThisMonth,
                'scheduled_reports' => $scheduledReports,
            ],
            'recentReports' => $recentReports,
        ];

        return view('reports/index', $data);
    }

    /**
     * Attendance Reports Page
     */
    public function attendance()
    {
        // Check permission
        if (!has_permission('generate-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to generate reports');
        }

        // Get filter dropdowns
        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();
        $employees = $this->employeeModel->getEmployeesWithFilters([]);

        $data = [
            'pageTitle' => 'Attendance Reports',
            'departments' => $departments,
            'locations' => $locations,
            'employees' => $employees,
        ];

        return view('reports/attendance', $data);
    }

    /**
     * Generate Attendance Report
     */
    public function generateAttendance()
    {
        // Check permission
        if (!has_permission('generate-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to generate reports');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'date_from' => 'required|valid_date',
            'date_to' => 'required|valid_date',
            'report_type' => 'required|in_list[summary,detailed,daily]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Get filters
        $filters = [
            'date_from' => $this->request->getPost('date_from'),
            'date_to' => $this->request->getPost('date_to'),
            'location_id' => $this->request->getPost('location_id'),
            'department_id' => $this->request->getPost('department_id'),
            'employee_id' => $this->request->getPost('employee_id'),
            'status' => $this->request->getPost('status'),
        ];

        $reportType = $this->request->getPost('report_type');
        $exportFormat = $this->request->getPost('export_format', FILTER_SANITIZE_SPECIAL_CHARS) ?? 'view';

        // Generate report data based on type
        switch ($reportType) {
            case 'summary':
                $reportData = $this->generateAttendanceSummary($filters);
                break;
            case 'detailed':
                $reportData = $this->generateAttendanceDetailed($filters);
                break;
            case 'daily':
                $reportData = $this->generateDailyAttendance($filters);
                break;
            default:
                $reportData = [];
        }

        // If export format is requested
        if ($exportFormat === 'pdf') {
            return $this->exportToPDF($reportData, $reportType, $filters);
        } elseif ($exportFormat === 'excel') {
            return $this->exportToExcel($reportData, $reportType, $filters);
        }

        // Otherwise display in browser
        $data = [
            'pageTitle' => 'Attendance Report - ' . ucfirst($reportType),
            'reportData' => $reportData,
            'reportType' => $reportType,
            'filters' => $filters,
            'departments' => $this->departmentModel->getActiveDepartments(),
            'locations' => $this->locationModel->getActiveLocations(),
            'employees' => $this->employeeModel->getEmployeesWithFilters([]),
        ];

        return view('reports/attendance', $data);
    }

    /**
     * Leave Reports Page
     */
    public function leave()
    {
        // Check permission
        if (!has_permission('generate-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to generate reports');
        }

        // Get filter dropdowns
        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();
        $employees = $this->employeeModel->getEmployeesWithFilters([]);

        $data = [
            'pageTitle' => 'Leave Reports',
            'departments' => $departments,
            'locations' => $locations,
            'employees' => $employees,
        ];

        return view('reports/leave', $data);
    }

    /**
     * Generate Leave Report
     */
    public function generateLeave()
    {
        // Check permission
        if (!has_permission('generate-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to generate reports');
        }

        // Validate input
        $validation = \Config\Services::validation();
        $validation->setRules([
            'date_from' => 'required|valid_date',
            'date_to' => 'required|valid_date',
            'report_type' => 'required|in_list[summary,balance,usage]',
        ]);

        if (!$validation->withRequest($this->request)->run()) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        // Get filters
        $filters = [
            'date_from' => $this->request->getPost('date_from'),
            'date_to' => $this->request->getPost('date_to'),
            'location_id' => $this->request->getPost('location_id'),
            'department_id' => $this->request->getPost('department_id'),
            'employee_id' => $this->request->getPost('employee_id'),
            'status' => $this->request->getPost('status'),
        ];

        $reportType = $this->request->getPost('report_type');
        $exportFormat = $this->request->getPost('export_format', FILTER_SANITIZE_SPECIAL_CHARS) ?? 'view';

        // Generate report data based on type
        switch ($reportType) {
            case 'summary':
                $reportData = $this->generateLeaveSummary($filters);
                break;
            case 'balance':
                $reportData = $this->generateLeaveBalance($filters);
                break;
            case 'usage':
                $reportData = $this->generateLeaveUsage($filters);
                break;
            default:
                $reportData = [];
        }

        // If export format is requested
        if ($exportFormat === 'pdf') {
            return $this->exportToPDF($reportData, $reportType, $filters);
        } elseif ($exportFormat === 'excel') {
            return $this->exportToExcel($reportData, $reportType, $filters);
        }

        // Otherwise display in browser
        $data = [
            'pageTitle' => 'Leave Report - ' . ucfirst($reportType),
            'reportData' => $reportData,
            'reportType' => $reportType,
            'filters' => $filters,
            'departments' => $this->departmentModel->getActiveDepartments(),
            'locations' => $this->locationModel->getActiveLocations(),
            'employees' => $this->employeeModel->getEmployeesWithFilters([]),
        ];

        return view('reports/leave', $data);
    }

    /**
     * Scheduled Reports Page
     */
    public function scheduled()
    {
        // Check permission
        if (!has_permission('schedule-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to schedule reports');
        }

        $db = \Config\Database::connect();
        $builder = $db->table('scheduled_reports');
        $builder->select('scheduled_reports.*, users.full_name');
        $builder->join('users', 'users.id = scheduled_reports.created_by', 'left');
        $builder->orderBy('scheduled_reports.created_at', 'DESC');
        $scheduledReports = $builder->get()->getResultArray();

        $data = [
            'pageTitle' => 'Scheduled Reports',
            'scheduledReports' => $scheduledReports,
        ];

        return view('reports/scheduled', $data);
    }

    /**
     * Export Report (handles PDF and Excel exports)
     */
    public function export($format)
    {
        // Check permission
        if (!has_permission('export-reports')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to export reports');
        }

        return redirect()->back()->with('info', 'Export functionality will be available soon');
    }

    // ===== PRIVATE HELPER METHODS =====

    /**
     * Generate Attendance Summary Report
     */
    private function generateAttendanceSummary($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('attendance a');

        $builder->select('
            e.employee_id,
            e.full_name,
            d.department_name,
            l.location_name,
            COUNT(CASE WHEN a.status = "present" THEN 1 END) as present_days,
            COUNT(CASE WHEN a.status = "absent" THEN 1 END) as absent_days,
            COUNT(CASE WHEN a.status = "late" THEN 1 END) as late_days,
            COUNT(CASE WHEN a.status = "half_day" THEN 1 END) as half_days,
            COUNT(*) as total_days
        ');

        $builder->join('employees e', 'e.employee_id = a.employee_id');
        $builder->join('departments d', 'd.id = e.department_id', 'left');
        $builder->join('locations l', 'l.id = e.location_id', 'left');

        $builder->where('a.date >=', $filters['date_from']);
        $builder->where('a.date <=', $filters['date_to']);

        if (!empty($filters['location_id'])) {
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('e.employee_id', $filters['employee_id']);
        }

        $builder->groupBy('e.employee_id');
        $builder->orderBy('e.full_name', 'ASC');

        return $builder->get()->getResultArray();
    }

    /**
     * Generate Detailed Attendance Report
     */
    private function generateAttendanceDetailed($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('attendance a');

        $builder->select('
            a.date,
            e.employee_id,
            e.full_name,
            d.department_name,
            l.location_name,
            a.check_in_time,
            a.check_out_time,
            a.status,
            a.remarks
        ');

        $builder->join('employees e', 'e.employee_id = a.employee_id');
        $builder->join('departments d', 'd.id = e.department_id', 'left');
        $builder->join('locations l', 'l.id = e.location_id', 'left');

        $builder->where('a.date >=', $filters['date_from']);
        $builder->where('a.date <=', $filters['date_to']);

        if (!empty($filters['location_id'])) {
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('e.employee_id', $filters['employee_id']);
        }

        if (!empty($filters['status'])) {
            $builder->where('a.status', $filters['status']);
        }

        $builder->orderBy('a.date', 'DESC');
        $builder->orderBy('e.full_name', 'ASC');

        return $builder->get()->getResultArray();
    }

    /**
     * Generate Daily Attendance Report
     */
    private function generateDailyAttendance($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('attendance a');

        $builder->select('
            a.date,
            COUNT(CASE WHEN a.status = "present" THEN 1 END) as present_count,
            COUNT(CASE WHEN a.status = "absent" THEN 1 END) as absent_count,
            COUNT(CASE WHEN a.status = "late" THEN 1 END) as late_count,
            COUNT(CASE WHEN a.status = "half_day" THEN 1 END) as half_day_count,
            COUNT(*) as total_count
        ');

        $builder->join('employees e', 'e.employee_id = a.employee_id');

        $builder->where('a.date >=', $filters['date_from']);
        $builder->where('a.date <=', $filters['date_to']);

        if (!empty($filters['location_id'])) {
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        $builder->groupBy('a.date');
        $builder->orderBy('a.date', 'ASC');

        return $builder->get()->getResultArray();
    }

    /**
     * Generate Leave Summary Report
     */
    private function generateLeaveSummary($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('leave_requests lr');

        $builder->select('
            e.employee_id,
            e.full_name,
            d.department_name,
            l.location_name,
            lt.leave_type_name,
            COUNT(lr.id) as total_requests,
            SUM(lr.total_days) as total_days,
            COUNT(CASE WHEN lr.status = "approved" THEN 1 END) as approved_count,
            COUNT(CASE WHEN lr.status = "rejected" THEN 1 END) as rejected_count,
            COUNT(CASE WHEN lr.status = "pending" THEN 1 END) as pending_count
        ');

        $builder->join('employees e', 'e.employee_id = lr.employee_id');
        $builder->join('departments d', 'd.id = e.department_id', 'left');
        $builder->join('locations l', 'l.id = e.location_id', 'left');
        $builder->join('leave_types lt', 'lt.id = lr.leave_type_id', 'left');

        $builder->where('lr.start_date >=', $filters['date_from']);
        $builder->where('lr.end_date <=', $filters['date_to']);

        if (!empty($filters['location_id'])) {
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('e.employee_id', $filters['employee_id']);
        }

        if (!empty($filters['status'])) {
            $builder->where('lr.status', $filters['status']);
        }

        $builder->groupBy('e.employee_id, lt.id');
        $builder->orderBy('e.full_name', 'ASC');

        return $builder->get()->getResultArray();
    }

    /**
     * Generate Leave Balance Report
     */
    private function generateLeaveBalance($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('leave_balances lb');

        $builder->select('
            e.employee_id,
            e.full_name,
            d.department_name,
            l.location_name,
            lt.leave_type_name,
            lb.allocated_days,
            lb.used_days,
            lb.remaining_days,
            lb.year
        ');

        $builder->join('employees e', 'e.employee_id = lb.employee_id');
        $builder->join('departments d', 'd.id = e.department_id', 'left');
        $builder->join('locations l', 'l.id = e.location_id', 'left');
        $builder->join('leave_types lt', 'lt.id = lb.leave_type_id', 'left');

        $builder->where('lb.year', date('Y'));

        if (!empty($filters['location_id'])) {
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('e.employee_id', $filters['employee_id']);
        }

        $builder->orderBy('e.full_name', 'ASC');
        $builder->orderBy('lt.leave_type_name', 'ASC');

        return $builder->get()->getResultArray();
    }

    /**
     * Generate Leave Usage Report
     */
    private function generateLeaveUsage($filters)
    {
        $db = \Config\Database::connect();
        $builder = $db->table('leave_requests lr');

        $builder->select('
            e.employee_id,
            e.full_name,
            d.department_name,
            lt.leave_type_name,
            lr.start_date,
            lr.end_date,
            lr.total_days,
            lr.status,
            lr.reason
        ');

        $builder->join('employees e', 'e.employee_id = lr.employee_id');
        $builder->join('departments d', 'd.id = e.department_id', 'left');
        $builder->join('leave_types lt', 'lt.id = lr.leave_type_id', 'left');

        $builder->where('lr.start_date >=', $filters['date_from']);
        $builder->where('lr.end_date <=', $filters['date_to']);

        if (!empty($filters['location_id'])) {
            $builder->join('locations l', 'l.id = e.location_id', 'left');
            $builder->where('e.location_id', $filters['location_id']);
        }

        if (!empty($filters['department_id'])) {
            $builder->where('e.department_id', $filters['department_id']);
        }

        if (!empty($filters['employee_id'])) {
            $builder->where('e.employee_id', $filters['employee_id']);
        }

        if (!empty($filters['status'])) {
            $builder->where('lr.status', $filters['status']);
        }

        $builder->orderBy('lr.start_date', 'DESC');

        return $builder->get()->getResultArray();
    }

    /**
     * Export Report to PDF
     */
    private function exportToPDF($data, $reportType, $filters)
    {
        // This would require a PDF library like TCPDF or Dompdf
        // Placeholder for future implementation
        return redirect()->back()->with('info', 'PDF export will be available soon');
    }

    /**
     * Export Report to Excel
     */
    private function exportToExcel($data, $reportType, $filters)
    {
        // This would require PhpSpreadsheet library
        // Placeholder for future implementation
        return redirect()->back()->with('info', 'Excel export will be available soon');
    }

    /**
     * Get Dashboard Statistics (API endpoint)
     */
    public function getDashboardStats()
    {
        // Set JSON response
        $this->response->setContentType('application/json');

        try {
            $db = \Config\Database::connect();
            $today = date('Y-m-d');
            $currentMonth = date('Y-m');

            // Get today's attendance rate
            $totalEmployees = $this->employeeModel->where('status', 'active')->countAllResults();
            $presentToday = $db->table('attendance')
                ->where('attendance_date', $today)
                ->where('status !=', 'absent')
                ->countAllResults();
            $attendanceRate = $totalEmployees > 0 ? round(($presentToday / $totalEmployees) * 100, 1) : 0;

            // Get pending leave requests
            $pendingLeaves = $this->leaveRequestModel
                ->where('status', 'pending')
                ->countAllResults();

            // Get reports generated this month
            $reportsThisMonth = $db->table('audit_logs')
                ->where('action', 'generate')
                ->where('module', 'reports')
                ->where('DATE_FORMAT(created_at, "%Y-%m")', $currentMonth)
                ->countAllResults();

            // Get scheduled reports count
            $scheduledReports = $db->table('scheduled_reports')
                ->where('is_active', 1)
                ->countAllResults(false);

            return $this->response->setStatusCode(200)->setJSON([
                'status' => 'success',
                'data' => [
                    'attendance_rate' => $attendanceRate,
                    'pending_leaves' => $pendingLeaves,
                    'reports_this_month' => $reportsThisMonth,
                    'scheduled_reports' => $scheduledReports,
                    'last_updated' => date('Y-m-d H:i:s')
                ]
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Dashboard stats error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Failed to fetch dashboard statistics'
            ]);
        }
    }
}
