<?php

namespace App\Controllers\Leave;

use App\Controllers\BaseController;
use App\Models\LeaveRequestModel;
use App\Models\LeaveTypeModel;
use App\Models\LeaveBalanceModel;
use App\Models\EmployeeModel;
use App\Models\LocationModel;
use App\Models\AuditLogModel;

class LeaveController extends BaseController
{
    protected $leaveRequestModel;
    protected $leaveTypeModel;
    protected $leaveBalanceModel;
    protected $employeeModel;
    protected $locationModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->leaveRequestModel = new LeaveRequestModel();
        $this->leaveTypeModel = new LeaveTypeModel();
        $this->leaveBalanceModel = new LeaveBalanceModel();
        $this->employeeModel = new EmployeeModel();
        $this->locationModel = new LocationModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display leave requests list
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-leave-requests')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view leave requests');
        }

        // Get filters
        $filters = [
            'status' => $this->request->getGet('status'),
            'leave_type_id' => $this->request->getGet('leave_type_id'),
            'employee_id' => $this->request->getGet('employee_id'),
            'location_id' => $this->request->getGet('location_id'),
            'date_from' => $this->request->getGet('date_from'),
            'date_to' => $this->request->getGet('date_to'),
        ];

        // Apply location filter if not super admin
        if (!is_super_admin() && !$filters['location_id']) {
            $filters['location_id'] = user_location_id();
        }

        // Get leave requests
        $requests = $this->leaveRequestModel->getRequestsWithDetails($filters);

        // Get leave types for filter
        $leaveTypes = $this->leaveTypeModel->getActiveLeaveTypes();

        // Get locations for filter
        $locations = $this->locationModel->getActiveLocations();

        // Get employees for filter
        $employees = $this->employeeModel->getEmployeesWithFilters(['status' => 'active']);

        // Get current user's employee info for approval validation
        $currentUserDepartmentId = null;
        $isManager = false;
        $userModel = new \App\Models\UserModel();
        $currentUser = $userModel->find(user_id());
        if ($currentUser && isset($currentUser['email'])) {
            // Find employee by matching email
            $currentEmployee = $this->employeeModel->where('email', $currentUser['email'])->first();
            if ($currentEmployee) {
                $currentUserDepartmentId = $currentEmployee['department_id'];
                $isManager = stripos($currentEmployee['position'], 'manager') !== false;
            }
        }

        $data = [
            'pageTitle' => 'Leave Requests',
            'requests' => $requests,
            'leaveTypes' => $leaveTypes,
            'locations' => $locations,
            'employees' => $employees,
            'filters' => $filters,
            'currentUserDepartmentId' => $currentUserDepartmentId,
            'isManager' => $isManager,
        ];

        return view('leave/index', $data);
    }

    /**
     * View single leave request
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('view-leave-requests')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view leave requests');
        }

        $request = $this->leaveRequestModel->select('leave_requests.*,
                employees.employee_number,
                employees.first_name,
                employees.last_name,
                employees.photo,
                employees.email,
                employees.phone,
                employees.department_id,
                leave_types.name as leave_type_name,
                leave_types.color,
                leave_types.is_paid,
                departments.name as department_name,
                locations.name as location_name,
                users.full_name as approved_by_name')
            ->join('employees', 'employees.employee_id = leave_requests.employee_id')
            ->join('leave_types', 'leave_types.id = leave_requests.leave_type_id')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = employees.location_id', 'left')
            ->join('users', 'users.id = leave_requests.approved_by', 'left')
            ->find($id);

        if (!$request) {
            return redirect()->to('/leave/requests')->with('error', 'Leave request not found');
        }

        // Get current user's employee info for approval validation
        $currentUserDepartmentId = null;
        $isManager = false;
        $userModel = new \App\Models\UserModel();
        $currentUser = $userModel->find(user_id());
        if ($currentUser && isset($currentUser['email'])) {
            // Find employee by matching email
            $currentEmployee = $this->employeeModel->where('email', $currentUser['email'])->first();
            if ($currentEmployee) {
                $currentUserDepartmentId = $currentEmployee['department_id'];
                $isManager = stripos($currentEmployee['position'], 'manager') !== false;
            }
        }

        $data = [
            'pageTitle' => 'Leave Request Details',
            'request' => $request,
            'currentUserDepartmentId' => $currentUserDepartmentId,
            'isManager' => $isManager,
        ];

        return view('leave/view', $data);
    }

    /**
     * Show create leave request form
     */
    public function create()
    {
        // Check permission
        if (!has_permission('create-leave-request')) {
            return redirect()->to('/leave/requests')->with('error', 'You do not have permission to create leave requests');
        }

        $leaveTypes = $this->leaveTypeModel->getActiveLeaveTypes();
        $employees = $this->employeeModel->getEmployeesWithFilters(['status' => 'active']);

        // Get current year balances for the logged-in user's employee record
        $currentYear = date('Y');

        $data = [
            'pageTitle' => 'Create Leave Request',
            'leaveTypes' => $leaveTypes,
            'employees' => $employees,
            'currentYear' => $currentYear,
        ];

        return view('leave/create', $data);
    }

    /**
     * Store leave request
     */
    public function store()
    {
        // Check permission
        if (!has_permission('create-leave-request')) {
            return redirect()->to('/leave/requests')->with('error', 'You do not have permission to create leave requests');
        }

        // Validation rules
        $rules = [
            'employee_id' => 'required|is_natural_no_zero',
            'leave_type_id' => 'required|is_natural_no_zero',
            'start_date' => 'required|valid_date',
            'end_date' => 'required|valid_date',
            'reason' => 'required|min_length[10]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Calculate total days
        $startDate = new \DateTime($this->request->getPost('start_date'));
        $endDate = new \DateTime($this->request->getPost('end_date'));
        $interval = $startDate->diff($endDate);
        $totalDays = $interval->days + 1; // Include both start and end dates

        // Prepare data
        $leaveData = [
            'employee_id' => $this->request->getPost('employee_id'),
            'leave_type_id' => $this->request->getPost('leave_type_id'),
            'start_date' => $this->request->getPost('start_date'),
            'end_date' => $this->request->getPost('end_date'),
            'total_days' => $totalDays,
            'reason' => $this->request->getPost('reason'),
            'status' => 'Pending',
        ];

        // Handle file upload
        $attachment = $this->request->getFile('attachment');
        if ($attachment && $attachment->isValid() && !$attachment->hasMoved()) {
            $newName = $attachment->getRandomName();
            $attachment->move(ROOTPATH . 'public/uploads/leave-documents', $newName);
            $leaveData['attachment'] = $newName;
        }

        // Insert leave request
        if ($this->leaveRequestModel->insert($leaveData)) {
            $requestId = $this->leaveRequestModel->getInsertID();

            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'create',
                    'leave_requests',
                    $requestId,
                    'Created leave request',
                    null,
                    json_encode($leaveData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/leave/requests')->with('success', 'Leave request submitted successfully');
        } else {
            $errors = $this->leaveRequestModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to submit leave request';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Approve leave request
     */
    public function approve($id)
    {
        // Check permission
        if (!has_permission('approve-leave')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to approve leave requests'
            ]);
        }

        // Get the leave request with employee details
        $leaveRequest = $this->leaveRequestModel->select('leave_requests.*,
                employees.department_id,
                employees.first_name,
                employees.last_name')
            ->join('employees', 'employees.employee_id = leave_requests.employee_id')
            ->find($id);

        if (!$leaveRequest) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Leave request not found'
            ]);
        }

        // Get the current user's employee record
        $userModel = new \App\Models\UserModel();
        $currentUser = $userModel->find(user_id());

        if (!$currentUser || !isset($currentUser['email'])) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Your user account is not linked to an employee record'
            ]);
        }

        // Find employee by matching email
        $approverEmployee = $this->employeeModel->where('email', $currentUser['email'])->first();

        if (!$approverEmployee) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Approver employee record not found'
            ]);
        }

        // Check if approver is a manager
        if (stripos($approverEmployee['position'], 'manager') === false && !is_super_admin()) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Only managers can approve leave requests'
            ]);
        }

        // Check if approver is in the same department (unless super admin)
        if (!is_super_admin() && $approverEmployee['department_id'] != $leaveRequest['department_id']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You can only approve leave requests from employees in your department'
            ]);
        }

        $comments = $this->request->getPost('comments');

        $result = $this->leaveRequestModel->approveRequest($id, user_id(), $comments);

        if ($result) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'approve',
                    'leave_requests',
                    $id,
                    'Approved leave request',
                    null,
                    json_encode(['comments' => $comments])
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Leave request approved successfully'
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to approve leave request'
            ]);
        }
    }

    /**
     * Reject leave request
     */
    public function reject($id)
    {
        // Check permission
        if (!has_permission('approve-leave')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to reject leave requests'
            ]);
        }

        // Get the leave request with employee details
        $leaveRequest = $this->leaveRequestModel->select('leave_requests.*,
                employees.department_id,
                employees.first_name,
                employees.last_name')
            ->join('employees', 'employees.employee_id = leave_requests.employee_id')
            ->find($id);

        if (!$leaveRequest) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Leave request not found'
            ]);
        }

        // Get the current user's employee record
        $userModel = new \App\Models\UserModel();
        $currentUser = $userModel->find(user_id());

        if (!$currentUser || !isset($currentUser['email'])) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Your user account is not linked to an employee record'
            ]);
        }

        // Find employee by matching email
        $approverEmployee = $this->employeeModel->where('email', $currentUser['email'])->first();

        if (!$approverEmployee) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Approver employee record not found'
            ]);
        }

        // Check if approver is a manager
        if (stripos($approverEmployee['position'], 'manager') === false && !is_super_admin()) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Only managers can reject leave requests'
            ]);
        }

        // Check if approver is in the same department (unless super admin)
        if (!is_super_admin() && $approverEmployee['department_id'] != $leaveRequest['department_id']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You can only reject leave requests from employees in your department'
            ]);
        }

        $rejectionReason = $this->request->getPost('rejection_reason');

        if (empty($rejectionReason)) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Rejection reason is required'
            ]);
        }

        $result = $this->leaveRequestModel->rejectRequest($id, user_id(), $rejectionReason);

        if ($result) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'reject',
                    'leave_requests',
                    $id,
                    'Rejected leave request',
                    null,
                    json_encode(['rejection_reason' => $rejectionReason])
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Leave request rejected successfully'
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to reject leave request'
            ]);
        }
    }

    /**
     * Display leave balances
     */
    public function balances()
    {
        // Check permission
        if (!has_permission('view-leave-balances')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view leave balances');
        }

        $selectedYear = $this->request->getGet('year') ?: date('Y');
        $selectedEmployeeId = $this->request->getGet('employee_id');

        // Get employees for filter
        $employees = $this->employeeModel->getEmployeesWithFilters(['status' => 'active']);

        // Get balances
        $balances = [];
        if ($selectedEmployeeId) {
            $balances = $this->leaveBalanceModel->getEmployeeBalances($selectedEmployeeId, $selectedYear);
        }

        $data = [
            'pageTitle' => 'Leave Balances',
            'employees' => $employees,
            'balances' => $balances,
            'selectedYear' => $selectedYear,
            'selectedEmployeeId' => $selectedEmployeeId,
        ];

        return view('leave/balances', $data);
    }

    /**
     * Display leave calendar
     */
    public function calendar()
    {
        // Check permission
        if (!has_permission('view-leave-requests')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view leave calendar');
        }

        $locationId = is_super_admin() ? null : user_location_id();

        // Get current month date range
        $startDate = date('Y-m-01');
        $endDate = date('Y-m-t');

        // Get leave calendar data
        $leaveCalendar = $this->leaveRequestModel->getLeaveCalendar($startDate, $endDate, $locationId);

        $data = [
            'pageTitle' => 'Leave Calendar',
            'leaveCalendar' => $leaveCalendar,
            'currentMonth' => date('F Y'),
        ];

        return view('leave/calendar', $data);
    }

    /**
     * Display leave types
     */
    public function types()
    {
        // Check permission
        if (!has_permission('manage-leave-types')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to manage leave types');
        }

        $leaveTypes = $this->leaveTypeModel->findAll();

        $data = [
            'pageTitle' => 'Leave Types',
            'leaveTypes' => $leaveTypes,
        ];

        return view('leave/types', $data);
    }
}
