<?php

namespace App\Controllers\Auth;

use App\Controllers\BaseController;
use App\Models\UserModel;
use App\Models\AuditLogModel;

class AuthController extends BaseController
{
    protected $userModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->auditLogModel = new AuditLogModel();
        helper(['form', 'url']);
    }

    /**
     * Display login page
     */
    public function login()
    {
        // Redirect to dashboard if already logged in
        if (session()->get('isLoggedIn')) {
            return redirect()->to('/dashboard');
        }

        return view('auth/login');
    }

    /**
     * Process login
     */
    public function attemptLogin()
    {
        
        $rules = [
            'username' => 'required|min_length[3]',
            'password' => 'required|min_length[6]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $username = $this->request->getPost('username');
        $password = $this->request->getPost('password');
        $remember = $this->request->getPost('remember_me');

        // Verify credentials
        $user = $this->userModel->verifyCredentials($username, $password);

        if (!$user) {
            // Log failed login attempt
            $this->auditLogModel->logAction(
                null,
                'Failed Login',
                'Authentication',
                null,
                "Failed login attempt for username: {$username}",
                null,
                null
            );

            return redirect()->back()
                ->withInput()
                ->with('error', 'Invalid username or password');
        }

        // Get user with role and permissions
        $userDetails = $this->userModel->getUserWithRole($user['id']);

        if (!$userDetails) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Unable to load user details');
        }

        // var_dump($userDetails); exit;
        // Set session data
        $sessionData = [
            'user_id'      => $userDetails['id'],
            'username'     => $userDetails['username'],
            'email'        => $userDetails['email'],
            'full_name'    => $userDetails['full_name'],
            'role_id'      => $userDetails['role_id'],
            'role_name'    => $userDetails['role_name'],
            'role_slug'    => $userDetails['role_slug'],
            'location_id'  => $userDetails['location_id'],
            'location_name' => $userDetails['location_name'],
            'permissions'  => $userDetails['permissions'],
            'profile_photo' => $userDetails['photo'],
            'isLoggedIn'   => true,
        ];

        session()->set($sessionData);

        // Set remember me cookie if requested
        if ($remember) {
            $this->response->setCookie('remember_token', $user['id'], 30 * 24 * 60 * 60); // 30 days
        }

        // Update last login
        $this->userModel->updateLastLogin($user['id']);

        // Log successful login
        $this->auditLogModel->logAction(
            $user['id'],
            'Login',
            'Authentication',
            null,
            "User logged in successfully",
            null,
            null
        );

        // Redirect to intended page or dashboard
        $redirect = session()->get('redirect_url') ?? '/dashboard';
        session()->remove('redirect_url');

        return redirect()->to($redirect)->with('success', 'Welcome back, ' . $userDetails['full_name'] . '!');
    }

    /**
     * Logout
     */
    public function logout()
    {
        $userId = session()->get('user_id');
        $username = session()->get('username');

        // Log logout
        if ($userId) {
            $this->auditLogModel->logAction(
                $userId,
                'Logout',
                'Authentication',
                null,
                "User logged out",
                null,
                null
            );
        }

        // Destroy session
        session()->destroy();

        // Remove remember me cookie
        $this->response->deleteCookie('remember_token');

        return redirect()->to('/login')->with('success', 'You have been logged out successfully');
    }

    /**
     * Display forgot password page
     */
    public function forgotPassword()
    {
        return view('auth/forgot_password');
    }

    /**
     * Send password reset link
     */
    public function sendResetLink()
    {
        $rules = [
            'email' => 'required|valid_email',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $email = $this->request->getPost('email');
        $user = $this->userModel->where('email', $email)->first();

        if (!$user) {
            // Don't reveal if email exists or not for security
            return redirect()->back()->with('success', 'If the email exists, a password reset link has been sent.');
        }

        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));

        $this->userModel->update($user['id'], [
            'password_reset_token'   => $token,
            'password_reset_expires' => $expires,
        ]);

        // TODO: Send email with reset link
        $resetLink = base_url("reset-password/{$token}");

        // Log password reset request
        $this->auditLogModel->logAction(
            $user['id'],
            'Password Reset Request',
            'Authentication',
            null,
            "Password reset requested for email: {$email}",
            null,
            null
        );

        // For now, just show success message
        return redirect()->back()->with('success', 'If the email exists, a password reset link has been sent.');
    }

    /**
     * Display reset password page
     */
    public function resetPassword($token = null)
    {
        if (!$token) {
            return redirect()->to('/login')->with('error', 'Invalid reset token');
        }

        // Verify token
        $user = $this->userModel
            ->where('password_reset_token', $token)
            ->where('password_reset_expires >', date('Y-m-d H:i:s'))
            ->first();

        if (!$user) {
            return redirect()->to('/login')->with('error', 'Invalid or expired reset token');
        }

        return view('auth/reset_password', ['token' => $token]);
    }

    /**
     * Process password reset
     */
    public function updatePassword()
    {
        $rules = [
            'token'            => 'required',
            'password'         => 'required|min_length[8]',
            'password_confirm' => 'required|matches[password]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $token = $this->request->getPost('token');
        $password = $this->request->getPost('password');

        // Verify token
        $user = $this->userModel
            ->where('password_reset_token', $token)
            ->where('password_reset_expires >', date('Y-m-d H:i:s'))
            ->first();

        if (!$user) {
            return redirect()->to('/login')->with('error', 'Invalid or expired reset token');
        }

        // Update password and clear reset token
        $this->userModel->update($user['id'], [
            'password_hash'          => $password, // Will be hashed by model callback
            'password_reset_token'   => null,
            'password_reset_expires' => null,
        ]);

        // Log password reset
        $this->auditLogModel->logAction(
            $user['id'],
            'Password Reset',
            'Authentication',
            null,
            "Password was reset successfully",
            null,
            null
        );

        return redirect()->to('/login')->with('success', 'Password reset successfully. Please login with your new password.');
    }

    /**
     * Display change password page
     */
    public function changePassword()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        return view('auth/change_password');
    }

    /**
     * Process password change
     */
    public function updateUserPassword()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $rules = [
            'current_password'  => 'required',
            'new_password'      => 'required|min_length[8]',
            'confirm_password'  => 'required|matches[new_password]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->with('errors', $this->validator->getErrors());
        }

        $userId = session()->get('user_id');
        $currentPassword = $this->request->getPost('current_password');
        $newPassword = $this->request->getPost('new_password');

        // Get user
        $user = $this->userModel->find($userId);

        // Verify current password
        if (!password_verify($currentPassword, $user['password_hash'])) {
            return redirect()->back()->with('error', 'Current password is incorrect');
        }

        // Update password
        $this->userModel->update($userId, [
            'password_hash' => $newPassword, // Will be hashed by model callback
        ]);

        // Log password change
        $this->auditLogModel->logAction(
            $userId,
            'Password Change',
            'Authentication',
            null,
            "User changed their password",
            null,
            null
        );

        return redirect()->back()->with('success', 'Password changed successfully');
    }
}
