<?php

namespace App\Controllers\Attendance;

use App\Controllers\BaseController;
use App\Models\AttendanceModel;
use App\Models\EmployeeModel;
use App\Models\DepartmentModel;
use App\Models\LocationModel;
use App\Models\AuditLogModel;

class AttendanceController extends BaseController
{
    protected $attendanceModel;
    protected $employeeModel;
    protected $departmentModel;
    protected $locationModel;
    protected $auditLogModel;

    public function __construct()
    {
        $this->attendanceModel = new AttendanceModel();
        $this->employeeModel = new EmployeeModel();
        $this->departmentModel = new DepartmentModel();
        $this->locationModel = new LocationModel();
        $this->auditLogModel = new AuditLogModel();

        helper('auth');
    }

    /**
     * Display attendance records (history)
     */
    public function index()
    {
        // Check permission
        if (!has_permission('view-attendance')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view attendance');
        }

        // Get filters from query string
        $filters = [
            'location_id' => $this->request->getGet('location_id'),
            'department_id' => $this->request->getGet('department_id'),
            'employee_id' => $this->request->getGet('employee_id'),
            'status' => $this->request->getGet('status'),
            'date_from' => $this->request->getGet('date_from') ?: date('Y-m-01'), // First day of current month
            'date_to' => $this->request->getGet('date_to') ?: date('Y-m-d'), // Today
        ];

        // Get attendance records
        $attendance = $this->attendanceModel->getAttendanceByDateRange(
            $filters['date_from'],
            $filters['date_to'],
            $filters
        );

        // Get filter dropdowns
        $departments = $this->departmentModel->getActiveDepartments();
        $locations = $this->locationModel->getActiveLocations();
        $employees = $this->employeeModel->getEmployeesWithFilters([]);

        $data = [
            'pageTitle' => 'Attendance Records',
            'attendance' => $attendance,
            'departments' => $departments,
            'locations' => $locations,
            'employees' => $employees,
            'filters' => $filters,
        ];

        return view('attendance/index', $data);
    }

    /**
     * Display today's attendance
     */
    public function today()
    {
        // Check permission
        if (!has_permission('view-attendance')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view attendance');
        }

        // Get location filter from query string
        $selectedLocationId = $this->request->getGet('location_id');

        // If user is not super admin and no location selected, use their location
        // If super admin and no location selected, show all locations
        if (!$selectedLocationId) {
            $locationId = is_super_admin() ? null : user_location_id();
        } else {
            // If user is not super admin, restrict to their location only
            if (!is_super_admin() && $selectedLocationId != user_location_id()) {
                $locationId = user_location_id();
            } else {
                $locationId = $selectedLocationId;
            }
        }

        // Get today's attendance
        $attendance = $this->attendanceModel->getTodayAttendance($locationId);

        // Get statistics
        $stats = $this->attendanceModel->getAttendanceStats(date('Y-m-d'), $locationId);

        // Build filters for employees
        $employeeFilters = ['status' => 'active'];
        if ($locationId) {
            $employeeFilters['location_id'] = $locationId;
        }

        // Get all active employees for the location(s)
        $allEmployees = $this->employeeModel->getEmployeesWithFilters($employeeFilters);

        // Find employees who haven't signed in yet
        $signedInEmployeeIds = array_column($attendance, 'employee_id');
        $notSignedIn = array_filter($allEmployees, function($emp) use ($signedInEmployeeIds) {
            return !in_array($emp['employee_id'], $signedInEmployeeIds);
        });

        // Calculate lunch break statistics
        $onLunch = 0; // Currently on lunch (lunch_out_time but no lunch_in_time)
        $completedLunch = 0; // Completed lunch (both lunch_out_time and lunch_in_time)
        $notOnLunch = 0; // Signed in but haven't gone to lunch yet

        foreach ($attendance as $record) {
            if ($record['lunch_out_time'] && $record['lunch_in_time']) {
                $completedLunch++;
            } elseif ($record['lunch_out_time']) {
                $onLunch++;
            } elseif (!$record['sign_out_time']) {
                $notOnLunch++;
            }
        }

        // Get all locations for filter dropdown
        $locations = $this->locationModel->getActiveLocations();

        // Get all departments for filter dropdown
        $departmentModel = new \App\Models\DepartmentModel();
        $departments = $departmentModel->getActiveDepartments();

        $data = [
            'pageTitle' => "Today's Attendance",
            'attendance' => $attendance,
            'stats' => $stats,
            'notSignedIn' => $notSignedIn,
            'totalEmployees' => count($allEmployees),
            'locations' => $locations,
            'departments' => $departments,
            'selectedLocationId' => $locationId,
            'lunchStats' => [
                'on_lunch' => $onLunch,
                'completed_lunch' => $completedLunch,
                'not_on_lunch' => $notOnLunch,
                'total_signed_in' => count($attendance),
            ],
        ];

        return view('attendance/today', $data);
    }

    /**
     * View single attendance record
     */
    public function view($id)
    {
        // Check permission
        if (!has_permission('view-attendance')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view attendance');
        }

        $attendance = $this->attendanceModel->select('attendance.*,
                employees.employee_number,
                employees.first_name,
                employees.last_name,
                employees.photo,
                employees.position,
                departments.name as department_name,
                locations.name as location_name')
            ->join('employees', 'employees.employee_id = attendance.employee_id')
            ->join('departments', 'departments.id = employees.department_id', 'left')
            ->join('locations', 'locations.id = attendance.location_id')
            ->find($id);

        if (!$attendance) {
            return redirect()->to('/attendance')->with('error', 'Attendance record not found');
        }

        $data = [
            'pageTitle' => 'Attendance Details',
            'attendance' => $attendance,
        ];

        return view('attendance/view', $data);
    }

    /**
     * Show manual entry form
     */
    public function manualEntry()
    {
        // Check permission
        if (!has_permission('manual-attendance')) {
            return redirect()->to('/attendance')->with('error', 'You do not have permission to manually enter attendance');
        }

        $locations = $this->locationModel->getActiveLocations();
        $employees = $this->employeeModel->getEmployeesWithFilters(['status' => 'active']);

        $data = [
            'pageTitle' => 'Manual Attendance Entry',
            'locations' => $locations,
            'employees' => $employees,
        ];

        return view('attendance/manual-entry', $data);
    }

    /**
     * Save manual attendance entry
     */
    public function saveManualEntry()
    {
        // Check permission
        if (!has_permission('manual-attendance')) {
            return redirect()->to('/attendance')->with('error', 'You do not have permission to manually enter attendance');
        }

        // Validation rules
        $rules = [
            'employee_id' => 'required|is_natural_no_zero',
            'attendance_date' => 'required|valid_date',
            'sign_in_time' => 'required',
            'location_id' => 'required|is_natural_no_zero',
            'status' => 'required|in_list[Present,Late,Absent,On Leave,Half Day,Work From Home]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Prepare data
        $attendanceData = [
            'employee_id' => $this->request->getPost('employee_id'),
            'location_id' => $this->request->getPost('location_id'),
            'attendance_date' => $this->request->getPost('attendance_date'),
            'sign_in_time' => $this->request->getPost('attendance_date') . ' ' . $this->request->getPost('sign_in_time'),
            'sign_out_time' => $this->request->getPost('sign_out_time') ?
                $this->request->getPost('attendance_date') . ' ' . $this->request->getPost('sign_out_time') : null,
            'sign_in_method' => 'Manual',
            'sign_out_method' => $this->request->getPost('sign_out_time') ? 'Manual' : null,
            'status' => $this->request->getPost('status'),
            'is_late' => $this->request->getPost('is_late') ? 1 : 0,
            'late_minutes' => $this->request->getPost('late_minutes') ?: 0,
            'notes' => $this->request->getPost('notes'),
            'created_by' => user_id(),
        ];

        // Calculate working hours if both sign in and sign out are provided
        if ($attendanceData['sign_out_time']) {
            $signIn = strtotime($attendanceData['sign_in_time']);
            $signOut = strtotime($attendanceData['sign_out_time']);
            $attendanceData['working_hours'] = round(($signOut - $signIn) / 3600, 2);
        }

        // Check if attendance already exists for this employee on this date
        $existing = $this->attendanceModel->where('employee_id', $attendanceData['employee_id'])
            ->where('attendance_date', $attendanceData['attendance_date'])
            ->first();

        if ($existing) {
            return redirect()->back()->withInput()->with('error', 'Attendance record already exists for this employee on this date');
        }

        // Insert attendance
        if ($this->attendanceModel->insert($attendanceData)) {
            $attendanceId = $this->attendanceModel->getInsertID();

            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'create',
                    'attendance',
                    $attendanceId,
                    'Manual attendance entry created',
                    null,
                    json_encode($attendanceData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log audit: ' . $e->getMessage());
            }

            return redirect()->to('/attendance/today')->with('success', 'Attendance recorded successfully');
        } else {
            $errors = $this->attendanceModel->errors();
            $errorMessage = !empty($errors) ? implode(', ', $errors) : 'Failed to record attendance';
            return redirect()->back()->withInput()->with('error', $errorMessage);
        }
    }

    /**
     * Show attendance corrections
     */
    public function corrections()
    {
        // Check permission
        if (!has_permission('view-attendance')) {
            return redirect()->to('/dashboard')->with('error', 'You do not have permission to view attendance corrections');
        }

        // Note: This would require an AttendanceCorrectionModel to be implemented
        // For now, return a placeholder view

        $data = [
            'pageTitle' => 'Attendance Corrections',
            'corrections' => [], // Placeholder
        ];

        return view('attendance/corrections', $data);
    }

    /**
     * Approve attendance correction
     */
    public function approveCorrection($id)
    {
        // Check permission
        if (!has_permission('approve-attendance-corrections')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to approve corrections'
            ]);
        }

        // Placeholder - would need AttendanceCorrectionModel
        return $this->response->setJSON([
            'success' => false,
            'message' => 'Correction approval feature is not yet implemented'
        ]);
    }

    /**
     * Reject attendance correction
     */
    public function rejectCorrection($id)
    {
        // Check permission
        if (!has_permission('approve-attendance-corrections')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to reject corrections'
            ]);
        }

        // Placeholder - would need AttendanceCorrectionModel
        return $this->response->setJSON([
            'success' => false,
            'message' => 'Correction rejection feature is not yet implemented'
        ]);
    }

    /**
     * Bulk sign-out for multiple employees
     */
    public function bulkSignOut()
    {
        // Check permission
        if (!has_permission('manage-attendance')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to manage attendance'
            ]);
        }

        // Get JSON input
        $json = $this->request->getJSON(true);
        $attendanceIds = $json['attendance_ids'] ?? [];

        if (empty($attendanceIds) || !is_array($attendanceIds)) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'No attendance records selected'
            ]);
        }

        $signOutTime = date('Y-m-d H:i:s');
        $successCount = 0;
        $errors = [];

        foreach ($attendanceIds as $attendanceId) {
            // Get attendance record
            $attendance = $this->attendanceModel->find($attendanceId);

            if (!$attendance) {
                $errors[] = "Attendance record ID {$attendanceId} not found";
                continue;
            }

            // Check if already signed out
            if ($attendance['sign_out_time']) {
                $errors[] = "Employee already signed out";
                continue;
            }

            // Calculate working hours
            $signInTime = strtotime($attendance['sign_in_time']);
            $signOutTimestamp = strtotime($signOutTime);
            $workingHours = round(($signOutTimestamp - $signInTime) / 3600, 2);

            // Update attendance record
            $updateData = [
                'sign_out_time' => $signOutTime,
                'sign_out_method' => 'Manual Bulk',
                'working_hours' => $workingHours,
            ];

            if ($this->attendanceModel->update($attendanceId, $updateData)) {
                $successCount++;

                // Log action
                try {
                    $this->auditLogModel->logAction(
                        user_id(),
                        'update',
                        'Attendance',
                        $attendanceId,
                        'Bulk sign-out for employee',
                        $attendance,
                        array_merge($attendance, $updateData)
                    );
                } catch (\Exception $e) {
                    log_message('error', 'Failed to log bulk sign-out audit: ' . $e->getMessage());
                }
            } else {
                $errors[] = "Failed to update attendance record ID {$attendanceId}";
            }
        }

        if ($successCount > 0) {
            return $this->response->setJSON([
                'success' => true,
                'message' => "Successfully signed out {$successCount} employee(s)" .
                            (!empty($errors) ? ". Some errors occurred: " . implode(', ', $errors) : '')
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to sign out employees: ' . implode(', ', $errors)
            ]);
        }
    }

    /**
     * Manual sign-out for a single employee from history
     */
    public function manualSignOut($id)
    {
        // Check permission
        if (!has_permission('manage-attendance')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to manage attendance'
            ]);
        }

        // Get attendance record
        $attendance = $this->attendanceModel->find($id);

        if (!$attendance) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Attendance record not found'
            ]);
        }

        // Check if already signed out
        if ($attendance['sign_out_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee has already signed out'
            ]);
        }

        // Calculate working hours
        $signOutTime = date('Y-m-d H:i:s');
        $signInTime = strtotime($attendance['sign_in_time']);
        $signOutTimestamp = strtotime($signOutTime);
        $workingHours = round(($signOutTimestamp - $signInTime) / 3600, 2);

        // Update attendance record
        $updateData = [
            'sign_out_time' => $signOutTime,
            'sign_out_method' => 'Manual',
            'working_hours' => $workingHours,
        ];

        if ($this->attendanceModel->update($id, $updateData)) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'update',
                    'Attendance',
                    $id,
                    'Manual sign-out by admin/manager',
                    $attendance,
                    array_merge($attendance, $updateData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log manual sign-out audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Employee signed out successfully'
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to update attendance record'
            ]);
        }
    }

    /**
     * Record lunch out
     */
    public function lunchOut($id)
    {
        // Check permission
        if (!has_permission('manage-attendance')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to manage attendance'
            ]);
        }

        // Get attendance record
        $attendance = $this->attendanceModel->find($id);

        if (!$attendance) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Attendance record not found'
            ]);
        }

        // Check if already lunch out
        if ($attendance['lunch_out_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Already marked lunch out'
            ]);
        }

        // Check if not signed in yet
        if (!$attendance['sign_in_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee must sign in first'
            ]);
        }

        // Check if already signed out
        if ($attendance['sign_out_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Employee has already signed out for the day'
            ]);
        }

        // Update attendance record
        $updateData = [
            'lunch_out_time' => date('Y-m-d H:i:s'),
            'lunch_out_method' => 'Manual',
        ];

        if ($this->attendanceModel->update($id, $updateData)) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'update',
                    'Attendance',
                    $id,
                    'Lunch out recorded',
                    $attendance,
                    array_merge($attendance, $updateData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log lunch out audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Lunch out recorded successfully'
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to record lunch out'
            ]);
        }
    }

    /**
     * Record lunch in
     */
    public function lunchIn($id)
    {
        // Check permission
        if (!has_permission('manage-attendance')) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'You do not have permission to manage attendance'
            ]);
        }

        // Get attendance record
        $attendance = $this->attendanceModel->find($id);

        if (!$attendance) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Attendance record not found'
            ]);
        }

        // Check if lunch out first
        if (!$attendance['lunch_out_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Must record lunch out first'
            ]);
        }

        // Check if already lunch in
        if ($attendance['lunch_in_time']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Already marked lunch in'
            ]);
        }

        // Calculate lunch duration
        $lunchOutTime = strtotime($attendance['lunch_out_time']);
        $lunchInTime = time();
        $lunchDuration = round(($lunchInTime - $lunchOutTime) / 60); // in minutes

        // Update attendance record
        $updateData = [
            'lunch_in_time' => date('Y-m-d H:i:s'),
            'lunch_in_method' => 'Manual',
            'lunch_duration_minutes' => $lunchDuration,
        ];

        if ($this->attendanceModel->update($id, $updateData)) {
            // Log action
            try {
                $this->auditLogModel->logAction(
                    user_id(),
                    'update',
                    'Attendance',
                    $id,
                    "Lunch in recorded (Duration: {$lunchDuration} minutes)",
                    $attendance,
                    array_merge($attendance, $updateData)
                );
            } catch (\Exception $e) {
                log_message('error', 'Failed to log lunch in audit: ' . $e->getMessage());
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => "Lunch in recorded successfully. Duration: {$lunchDuration} minutes"
            ]);
        } else {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Failed to record lunch in'
            ]);
        }
    }
}
