<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use App\Models\EmployeeModel;

class EmployeeController extends BaseController
{
    protected $employeeModel;

    public function __construct()
    {
        $this->employeeModel = new EmployeeModel();
    }

    /**
     * Get list of all employees
     *
     * GET /api/employees
     */
    public function index()
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            // Get all active employees with their department and location information
            $employees = $this->employeeModel
                ->select('
                    employees.employee_id,
                    employees.employee_number,
                    employees.qr_code,
                    employees.first_name,
                    employees.last_name,
                    employees.status,
                    departments.name,
                    locations.name
                ')
                ->join('departments', 'departments.id = employees.department_id', 'left')
                ->join('locations', 'locations.id = employees.location_id', 'left')
                ->orderBy('employees.employee_number', 'ASC')
                ->findAll();

            // Format the response data
            $formattedEmployees = array_map(function($employee) {
                return [
                    'employee_id' => $employee['employee_id'],
                    'employee_number' => $employee['employee_number'],
                    'qr_code' => $employee['qr_code'],
                    'full_name' => trim($employee['first_name'] . ' ' . $employee['last_name']),
                    'department' => $employee['name'] ?? 'N/A',
                    'location' => $employee['name'] ?? 'N/A',
                    'account_status' => $employee['status']
                ];
            }, $employees);

            return $this->response->setStatusCode(200)->setJSON([
                'status' => 'success',
                'total' => count($formattedEmployees),
                'data' => $formattedEmployees
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Employees API error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred while fetching employees'
            ]);
        }
    }

    /**
     * Get a single employee by employee number or QR code
     *
     * GET /api/employees/{identifier}
     */
    public function show($identifier)
    {
        // Set JSON response header
        $this->response->setContentType('application/json');

        try {
            // Try to find employee by employee_number or qr_code
            $employee = $this->employeeModel
                ->select('
                    employees.employee_id,
                    employees.employee_number,
                    employees.qr_code,
                    employees.first_name,
                    employees.last_name,
                    employees.email,
                    employees.phone,
                    employees.status,
                    departments.name,
                    locations.name
                ')
                ->join('departments', 'departments.id = employees.department_id', 'left')
                ->join('locations', 'locations.id = employees.location_id', 'left')
                ->groupStart()
                    ->where('employees.employee_number', $identifier)
                    ->orWhere('employees.qr_code', $identifier)
                ->groupEnd()
                ->first();

            if (!$employee) {
                return $this->response->setStatusCode(404)->setJSON([
                    'status' => 'error',
                    'message' => 'Employee not found'
                ]);
            }

            // Format the response data
            $formattedEmployee = [
                'employee_number' => $employee['employee_number'],
                'qr_code' => $employee['qr_code'],
                'full_name' => trim($employee['first_name'] . ' ' . $employee['last_name']),
                'email' => $employee['email'],
                'phone' => $employee['phone'],
                'department' => $employee['name'] ?? 'N/A',
                'location' => $employee['name'] ?? 'N/A',
                'account_status' => $employee['status']
            ];

            return $this->response->setStatusCode(200)->setJSON([
                'status' => 'success',
                'data' => $formattedEmployee
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Employee API error: ' . $e->getMessage());

            return $this->response->setStatusCode(500)->setJSON([
                'status' => 'error',
                'message' => 'Server error occurred while fetching employee'
            ]);
        }
    }
}
