# Employees API Endpoint Documentation

## Overview
API endpoints to retrieve employee information for mobile app integration.

---

## Endpoints

### 1. Get All Employees

Retrieve a list of all employees with their basic information.

**Endpoint:** `GET /api/employees`

**Request Headers:**
```
Content-Type: application/json
```

**Response (HTTP 200):**
```json
{
  "status": "success",
  "total": 150,
  "data": [
    {
      "employee_number": "EMP001",
      "qr_code": "QR123456789",
      "full_name": "John Doe",
      "department": "IT Department",
      "location": "Main Office",
      "account_status": "Active"
    },
    {
      "employee_number": "EMP002",
      "qr_code": "QR987654321",
      "full_name": "Jane Smith",
      "department": "HR Department",
      "location": "Branch Office",
      "account_status": "Active"
    }
  ]
}
```

**Error Response (HTTP 500):**
```json
{
  "status": "error",
  "message": "Server error occurred while fetching employees"
}
```

---

### 2. Get Single Employee

Retrieve detailed information about a specific employee by employee number or QR code.

**Endpoint:** `GET /api/employees/{identifier}`

**URL Parameters:**
- `identifier` - Can be either `employee_number` or `qr_code`

**Examples:**
- `GET /api/employees/EMP001`
- `GET /api/employees/QR123456789`

**Response (HTTP 200):**
```json
{
  "status": "success",
  "data": {
    "employee_number": "EMP001",
    "qr_code": "QR123456789",
    "full_name": "John Doe",
    "email": "john.doe@company.com",
    "phone": "0712345678",
    "department": "IT Department",
    "location": "Main Office",
    "account_status": "Active"
  }
}
```

**Error Response - Not Found (HTTP 404):**
```json
{
  "status": "error",
  "message": "Employee not found"
}
```

**Error Response - Server Error (HTTP 500):**
```json
{
  "status": "error",
  "message": "Server error occurred while fetching employee"
}
```

---

## Response Fields

### All Employees Endpoint

| Field | Type | Description |
|-------|------|-------------|
| `employee_number` | string | Unique employee number |
| `qr_code` | string | Employee's QR code for attendance |
| `full_name` | string | Employee's full name (first + last) |
| `department` | string | Department name (or "N/A" if not assigned) |
| `location` | string | Work location name (or "N/A" if not assigned) |
| `account_status` | string | Account status (Active/Inactive) |

### Single Employee Endpoint

Includes all fields from the list endpoint, plus:

| Field | Type | Description |
|-------|------|-------------|
| `email` | string | Employee's email address |
| `phone` | string | Employee's phone number |

---

## Testing

### Using cURL

**Get All Employees:**
```bash
curl -X GET http://localhost/real-time-signin/public/api/employees \
  -H "Content-Type: application/json"
```

**Get Single Employee:**
```bash
curl -X GET http://localhost/real-time-signin/public/api/employees/EMP001 \
  -H "Content-Type: application/json"
```

### Using Postman

**Get All Employees:**
1. Method: GET
2. URL: `http://localhost/real-time-signin/public/api/employees`
3. Headers: `Content-Type: application/json`

**Get Single Employee:**
1. Method: GET
2. URL: `http://localhost/real-time-signin/public/api/employees/EMP001`
3. Headers: `Content-Type: application/json`

---

## Implementation Notes

### Controller Location
`app/Controllers/Api/EmployeeController.php`

### Key Features

1. **Joins with Related Tables**
   - Joins with `departments` table to get department name
   - Joins with `locations` table to get location name
   - Uses LEFT JOIN to include employees without department/location

2. **Flexible Search**
   - Single employee endpoint accepts both employee_number and qr_code
   - Automatically detects which field to search

3. **Data Formatting**
   - Combines first_name and last_name into full_name
   - Returns "N/A" for null department/location values
   - Ordered by employee_number ascending

4. **Error Handling**
   - Logs errors to system log
   - Returns appropriate HTTP status codes
   - Provides user-friendly error messages

---

## Security Notes

**IMPORTANT:** These endpoints currently have NO authentication. For production use:

1. Add API key authentication
2. Implement rate limiting
3. Add IP whitelisting (if needed)
4. Ensure HTTPS is used
5. Consider implementing pagination for large datasets

---

## Routes Configuration

Located in `app/Config/Routes.php`:

```php
$routes->group('api', function($routes) {
    // Employee endpoints
    $routes->get('employees', 'Api\EmployeeController::index');
    $routes->get('employees/(:segment)', 'Api\EmployeeController::show/$1');
});
```

These routes are already excluded from CSRF protection via the Filters configuration.
